<?php

declare(strict_types=1);

namespace Drupal\Tests\supported_image\FunctionalJavascript;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\node\Entity\Node;
use Drupal\Tests\image\FunctionalJavascript\ImageFieldWidgetMultipleTest;
use Drupal\Tests\supported_image\Kernel\SupportedImageFieldCreationTrait;

/**
 * Tests the supported_image field widget support multiple upload correctly.
 *
 * Adapted from Drupal core's ImageFieldWidgetMultipleTest.
 *
 * @see \Drupal\Tests\image\FunctionalJavascript\ImageFieldWidgetMultipleTest
 * @group supported_image
 */
class SupportedImageFieldWidgetTest extends ImageFieldWidgetMultipleTest {

  use SupportedImageFieldCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['filter', 'text', 'supported_image'];

  /**
   * Tests the widget element.
   */
  public function testWidgetElement(): void {
    $file_system = \Drupal::service('file_system');
    $web_driver = $this->getSession()->getDriver();

    $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);
    $field_name = 'image';
    $field_settings = ['alt_field_required' => 0];
    $this
      ->createSupportedImageField($field_name, 'node', 'article', [], $field_settings);
    $this
      ->drupalLogin($this->drupalCreateUser(['access content', 'create article content']));
    $this->drupalGet('node/add/article');

    // Verify the widget does not contain the caption or attribution fields.
    // These should only show once a file is uploaded.
    $this->assertSession()->pageTextNotContains('Caption');
    $this->assertSession()->pageTextNotContains('Attribution');

    $image = $this->getTestFiles('image')[0];
    $path = $file_system->realpath($image->uri);
    $remote_path = $web_driver->uploadFileAndGetRemoteFilePath($path);

    // Insert an image and wait for the preview to show.
    $field_element = $this
      ->assertSession()
      ->elementExists('css', 'input.form-file');
    $field_element->setValue($remote_path);
    $this
      ->assertSession()
      ->waitForElementVisible('css', '[data-drupal-selector="edit-image-0-preview"]');

    // Verify the widget contains the caption field.
    $this->assertSession()->pageTextContains('Caption');
    $this
      ->assertSession()
      ->pageTextContains('A small amount of text that identifies subjects, provides context, or offers additional information about the image.');

    // Verify the widget contains the attribution field.
    $this->assertSession()->pageTextContains('Attribution');
    $this
      ->assertSession()
      ->pageTextContains("Credit the author or copyright holder of the image, as may be required by the image's license.");

    // Require the caption field and verify that the field is required.
    $field_config = FieldConfig::loadByName('node', 'article', $field_name);
    $field_config->setSetting('caption_field_required', TRUE)->save();
    $this->drupalGet('node/add/article');
    // Insert an image and wait for the preview to show.
    $field_element = $this
      ->assertSession()
      ->elementExists('css', 'input.form-file');
    $field_element->setValue($remote_path);
    $this
      ->assertSession()
      ->waitForElementVisible('css', '[data-drupal-selector="edit-image-0-preview"]');
    $this
      ->assertSession()
      ->elementExists('css', '.field--widget-supported-image-image .form-item[class$=caption-value] .form-required');

    // Require the attribution field and verify that the field is required.
    $field_config = FieldConfig::loadByName('node', 'article', $field_name);
    $field_config->setSetting('attribution_field_required', TRUE)->save();
    $this->drupalGet('node/add/article');
    // Insert an image and wait for the preview to show.
    $field_element = $this
      ->assertSession()
      ->elementExists('css', 'input.form-file');
    $field_element->setValue($remote_path);
    $this
      ->assertSession()
      ->waitForElementVisible('css', '[data-drupal-selector="edit-image-0-preview"]');
    $this
      ->assertSession()
      ->elementExists('css', '.field--widget-supported-image-image .form-item[class$=attribution-value] .form-required');

    // Set the field to not collect caption and verify caption is no longer
    // present in the widget.
    $field_config = FieldConfig::loadByName('node', 'article', $field_name);
    $field_config
      ->setSetting('caption_field', FALSE)
      ->setSetting('caption_field_required', FALSE)
      ->save();
    $this->drupalGet('node/add/article');
    // Insert an image and wait for the preview to show.
    $field_element = $this
      ->assertSession()
      ->elementExists('css', 'input.form-file');
    $field_element->setValue($remote_path);
    $this
      ->assertSession()
      ->waitForElementVisible('css', '[data-drupal-selector="edit-image-0-preview"]');
    $this->assertSession()->pageTextNotContains('Caption');

    // Set the field to not collect attribution and verify attribution is no
    // longer present in the widget.
    $field_config = FieldConfig::loadByName('node', 'article', $field_name);
    $field_config
      ->setSetting('attribution_field', FALSE)
      ->setSetting('attribution_field_required', FALSE)
      ->save();
    $this->drupalGet('node/add/article');
    // Insert an image and wait for the preview to show.
    $field_element = $this
      ->assertSession()
      ->elementExists('css', 'input.form-file');
    $field_element->setValue($remote_path);
    $this
      ->assertSession()
      ->waitForElementVisible('css', '[data-drupal-selector="edit-image-0-preview"]');
    $this->assertSession()->pageTextNotContains('Attribution');
  }

  /**
   * Tests that the widget supports multiple upload.
   */
  public function testWidgetElementMultipleUploads(): void {
    $image_factory = \Drupal::service('image.factory');
    $file_system = \Drupal::service('file_system');
    $web_driver = $this->getSession()->getDriver();

    $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);
    $field_name = 'images';
    $storage_settings = ['cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED];
    $field_settings = ['alt_field_required' => 0];
    $this->createSupportedImageField($field_name, 'node', 'article', $storage_settings, $field_settings);
    $this->drupalLogin($this->drupalCreateUser(['access content', 'create article content']));
    $this->drupalGet('node/add/article');
    $this->assertSession()->fieldExists('title[0][value]')->setValue('Test');

    $images = $this->getTestFiles('image');
    $images = array_slice($images, 0, 5);

    $paths = [];
    foreach ($images as $image) {
      $paths[] = $file_system->realpath($image->uri);
    }

    $remote_paths = [];
    foreach ($paths as $path) {
      $remote_paths[] = $web_driver->uploadFileAndGetRemoteFilePath($path);
    }

    $multiple_field = $this->assertSession()->elementExists('xpath', '//input[@multiple]');
    $multiple_field->setValue(implode("\n", $remote_paths));
    $this->assertSession()->waitForElementVisible('css', '[data-drupal-selector="edit-images-4-preview"]');
    $this->getSession()->getPage()->findButton('Save')->click();
    $this
      ->assertSession()
      ->waitForElementVisible('css', 'div[data-drupal-messages] a[href="/node/1"]');

    $node = Node::load(1);
    foreach ($paths as $delta => $path) {
      $node_image = $node->{$field_name}[$delta];
      $original_image = $image_factory->get($path);
      $this->assertEquals($original_image->getWidth(), $node_image->width, "Correct width of image #$delta");
      $this->assertEquals($original_image->getHeight(), $node_image->height, "Correct height of image #$delta");
    }
  }

}
