<?php

declare(strict_types=1);

namespace Drupal\Tests\supported_image\Kernel;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\field\Kernel\FieldKernelTestBase;
use Drupal\Tests\image\Kernel\ImageFormatterTest;

/**
 * Tests supported_image field rendering using entity fields.
 *
 * Adapted from Drupal\Tests\image\Kernel\ImageFormatterTest.
 *
 * @see \Drupal\Tests\image\Kernel\ImageFormatterTest
 * @group supported_image
 */
class SupportedImageFormatterTest extends ImageFormatterTest {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['filter', 'text', 'supported_image'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    FieldKernelTestBase::setUp();

    $this->installConfig(['field', 'filter', 'text']);
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('file');
    $this->installEntitySchema('filter_format');
    $this->installSchema('file', ['file_usage']);

    $this->entityType = 'entity_test';
    $this->bundle = $this->entityType;
    $this->fieldName = $this->randomMachineName();

    FieldStorageConfig::create([
      'entity_type' => $this->entityType,
      'field_name' => $this->fieldName,
      'type' => 'supported_image',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
    ])->save();
    FieldConfig::create([
      'entity_type' => $this->entityType,
      'field_name' => $this->fieldName,
      'bundle' => $this->bundle,
      'settings' => [
        'file_extensions' => 'jpg',
      ],
    ])->save();

    $this->display = \Drupal::service('entity_display.repository')
      ->getViewDisplay($this->entityType, $this->bundle)
      ->setComponent($this->fieldName, [
        'type' => 'supported_image',
        'label' => 'hidden',
      ]);
    $this->display->save();
  }

  /**
   * Tests supported_image formatter expected content output.
   */
  public function testSupportedImageFormatterExpectedContent(): void {
    // Create a test entity with the image field set.
    $entity = EntityTest::create([
      'name' => $this->randomMachineName(),
    ]);
    $entity->{$this->fieldName}->generateSampleItems(2);
    $entity->save();

    // Generate the render array to verify caption and attribution are present.
    $build = $this->display->build($entity);

    /** @var \Drupal\Core\Render\RendererInterface $renderer */
    $renderer = $this->container->get('renderer');

    $output = $renderer->renderRoot($build[$this->fieldName][0]);
    $this->assertStringContainsString('<figure class="supported-image">', (string) $output);
    $this->assertStringContainsString('<img', (string) $output);
    $this->assertStringContainsString('<figcaption>', (string) $output);
    $this->assertStringContainsString('<div class="caption">', (string) $output);
    $this->assertStringContainsString('<div class="attribution">', (string) $output);

    // Turn caption display off and see that caption no longer displays.
    $this->display->setComponent($this->fieldName, [
      'type' => 'supported_image',
      'label' => 'hidden',
      'settings' => [
        'display_caption' => FALSE,
      ],
    ])->save();
    $build = $this->display->build($entity);
    $output = $renderer->renderRoot($build[$this->fieldName][0]);
    $this->assertStringContainsString('<figure class="supported-image">', (string) $output);
    $this->assertStringContainsString('<img', (string) $output);
    $this->assertStringContainsString('<figcaption>', (string) $output);
    $this->assertStringNotContainsString('<div class="caption">', (string) $output);
    $this->assertStringContainsString('<div class="attribution">', (string) $output);

    // Turn attribution display off also and see neither caption or attribution
    // are shown.
    // The figcaption should also be gone now that both caption and attribution
    // are not displayed.
    $this->display->setComponent($this->fieldName, [
      'type' => 'supported_image',
      'label' => 'hidden',
      'settings' => [
        'display_caption' => FALSE,
        'display_attribution' => FALSE,
      ],
    ])->save();
    $build = $this->display->build($entity);
    $output = $renderer->renderRoot($build[$this->fieldName][0]);
    $this->assertStringContainsString('<figure class="supported-image">', (string) $output);
    $this->assertStringContainsString('<img', (string) $output);
    $this->assertStringNotContainsString('<figcaption>', (string) $output);
    $this->assertStringNotContainsString('<div class="caption">', (string) $output);
    $this->assertStringNotContainsString('<div class="attribution">', (string) $output);

    // Turn caption and attribution display back on and see they are present
    // in the output again.
    $this->display->setComponent($this->fieldName, [
      'type' => 'supported_image',
      'label' => 'hidden',
      'settings' => [
        'display_caption' => TRUE,
        'display_attribution' => TRUE,
      ],
    ])->save();
    $build = $this->display->build($entity);
    $output = $renderer->renderRoot($build[$this->fieldName][0]);
    $this->assertStringContainsString('<figure class="supported-image">', (string) $output);
    $this->assertStringContainsString('<img', (string) $output);
    $this->assertStringContainsString('<figcaption>', (string) $output);
    $this->assertStringContainsString('<div class="caption">', (string) $output);
    $this->assertStringContainsString('<div class="attribution">', (string) $output);

    // Clear caption value and see that it is gone from output.
    $entity->get($this->fieldName)->first()->setValue([
      'caption_value' => NULL,
    ] + $entity->get($this->fieldName)->first()->getValue());
    $entity->save();
    $build = $this->display->build($entity);
    $output = $renderer->renderRoot($build[$this->fieldName][0]);
    $this->assertStringContainsString('<figure class="supported-image">', (string) $output);
    $this->assertStringContainsString('<img', (string) $output);
    $this->assertStringContainsString('<figcaption>', (string) $output);
    $this->assertStringNotContainsString('<div class="caption">', (string) $output);
    $this->assertStringContainsString('<div class="attribution">', (string) $output);

    // Clear attribution value and see that it is gone from output.
    // The figcaption should also be gone now that both caption and attribution
    // are empty.
    $entity->get($this->fieldName)->first()->setValue([
      'attribution_value' => NULL,
    ] + $entity->get($this->fieldName)->first()->getValue());
    $entity->save();
    $build = $this->display->build($entity);
    $output = $renderer->renderRoot($build[$this->fieldName][0]);
    $this->assertStringContainsString('<figure class="supported-image">', (string) $output);
    $this->assertStringContainsString('<img', (string) $output);
    $this->assertStringNotContainsString('<figcaption>', (string) $output);
    $this->assertStringNotContainsString('<div class="caption">', (string) $output);
    $this->assertStringNotContainsString('<div class="attribution">', (string) $output);
  }

}
