<?php

declare(strict_types=1);

namespace Drupal\Tests\supported_image\Kernel;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\Url;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\file\Entity\File;
use Drupal\image\Entity\ImageStyle;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\image\Kernel\ImageThemeFunctionTest;

/**
 * Tests supported_image theme functions.
 *
 * Adapted from Drupal\Tests\image\Kernel\ImageThemeFunctionTest.
 *
 * @see \Drupal\Tests\image\Kernel\ImageThemeFunctionTest
 * @group supported_image
 */
class SupportedImageThemeFunctionTest extends ImageThemeFunctionTest {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['filter', 'text', 'supported_image'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    KernelTestBase::setUp();

    $this->installConfig(['field', 'filter', 'text']);
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('file');
    $this->installEntitySchema('filter_format');
    $this->installEntitySchema('user');
    $this->installSchema('file', ['file_usage']);

    FieldStorageConfig::create([
      'entity_type' => 'entity_test',
      'field_name' => 'image_test',
      'type' => 'supported_image',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
    ])->save();
    FieldConfig::create([
      'entity_type' => 'entity_test',
      'field_name' => 'image_test',
      'bundle' => 'entity_test',
    ])->save();
    \Drupal::service('file_system')->copy($this->root . '/core/misc/druplicon.png', 'public://example.jpg');
    $this->image = File::create([
      'uri' => 'public://example.jpg',
    ]);
    $this->image->save();
    $this->imageFactory = $this->container->get('image.factory');
  }

  /**
   * Tests usage of the supported_image field formatters.
   */
  public function testImageFormatterTheme(): void {
    /** @var \Drupal\Core\Render\RendererInterface $renderer */
    $renderer = $this->container->get('renderer');

    // Create an image.
    $files = $this->drupalGetTestFiles('image');
    $file = reset($files);
    $original_uri = \Drupal::service('file_system')->copy($file->uri, 'public://', FileExists::Rename);

    // Create a style.
    $style = ImageStyle::create(['name' => 'test', 'label' => 'Test']);
    $style->save();
    $url = \Drupal::service('file_url_generator')->transformRelative($style->buildUrl($original_uri));

    // Create a test entity with the image field set.
    $entity = EntityTest::create();
    $entity->image_test->target_id = $this->image->id();
    $entity->image_test->alt = NULL;
    $entity->image_test->uri = $original_uri;
    $image = $this->imageFactory->get('public://example.jpg');
    $entity->save();

    // Create the base element that we'll use in the tests below.
    $path = $this->randomMachineName();
    $base_element = [
      '#theme' => 'supported_image_formatter',
      '#image_style' => 'test',
      '#item' => $entity->image_test,
      '#url' => Url::fromUri('base:' . $path),
    ];

    // Test using theme_supported_image_formatter() with a NULL value for the
    // alt option.
    $element = $base_element;
    $this->setRawContent($renderer->renderRoot($element));
    $elements = $this
      ->xpath('//a[@href=:path]/img[@src=:url and @width=:width and @height=:height]', [
        ':path' => base_path() . $path,
        ':url' => $url,
        ':width' => $image->getWidth(),
        ':height' => $image->getHeight(),
      ]);
    $this->assertCount(1, $elements);

    // Test using theme_image_formatter() without an image title, alt text, or
    // link options.
    $element = $base_element;
    $element['#item']->alt = '';
    $this->setRawContent($renderer->renderRoot($element));
    $elements = $this
      ->xpath('//a[@href=:path]/img[@src=:url and @width=:width and @height=:height and @alt=""]', [
        ':path' => base_path() . $path,
        ':url' => $url,
        ':width' => $image->getWidth(),
        ':height' => $image->getHeight(),
      ]);
    $this->assertCount(1, $elements);

    // Link the image to a fragment on the page, and not a full URL.
    $fragment = $this->randomMachineName();
    $element = $base_element;
    $element['#url'] = Url::fromRoute('<none>', [], ['fragment' => $fragment]);
    $this->setRawContent($renderer->renderRoot($element));
    $elements = $this->xpath('//a[@href=:fragment]/img[@src=:url and @width=:width and @height=:height and @alt=""]', [
      ':fragment' => '#' . $fragment,
      ':url' => $url,
      ':width' => $image->getWidth(),
      ':height' => $image->getHeight(),
    ]);
    $this->assertCount(1, $elements);

    // Add caption content and see that it is rendered.
    $element = $base_element;
    $element['#item']->caption_value = $this->randomMachineName();
    $element['#item']->caption_format = filter_fallback_format();
    $this->setRawContent($renderer->renderRoot($element));
    $elements = $this->xpath('//figure[@class="supported-image"]/figcaption/div[@class="caption"]/*[contains(text(), :caption_value)]', [
      ':caption_value' => $element['#item']->caption_value,
    ]);
    $this->assertCount(1, $elements);

    // Add caption and attribution content and see that they are both rendered.
    $element = $base_element;
    $element['#item']->caption_value = $this->randomMachineName();
    $element['#item']->caption_format = filter_fallback_format();
    $element['#item']->attribution_value = $this->randomMachineName();
    $element['#item']->attribution_format = filter_fallback_format();
    $this->setRawContent($renderer->renderRoot($element));
    $elements = $this->xpath('//figure[@class="supported-image"]/figcaption/div[@class="caption"]/*[contains(text(), :caption_value)]', [
      ':caption_value' => $element['#item']->caption_value,
    ]);
    $this->assertCount(1, $elements);
    $elements = $this->xpath('//figure[@class="supported-image"]/figcaption/div[@class="attribution"]/*[contains(text(), :attribution_value)]', [
      ':attribution_value' => $element['#item']->attribution_value,
    ]);
    $this->assertCount(1, $elements);

    // Remove supporting caption and attribution content and see that
    // the figure is still present, but the figcaption is gone.
    $element = $base_element;
    $element['#item']->caption_value = NULL;
    $element['#item']->caption_format = NULL;
    $element['#item']->attribution_value = NULL;
    $element['#item']->attribution_format = NULL;
    $this->setRawContent($renderer->renderRoot($element));
    $elements = $this->xpath('//figure[@class="supported-image"]');
    $this->assertCount(1, $elements);
    $elements = $this->xpath('//figure[@class="supported-image"]/figcaption');
    $this->assertCount(0, $elements);
  }

}
