<?php

namespace Drupal\Surge\Console\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Console\Input\InputOption;

class ScanGuidelinesCommand extends Command
{
    protected static $defaultName = 'scan-guidelines';
    protected static $defaultDescription = 'Scan for community-contributed guidelines and merge them with main standards';

    private $projectRoot;
    private $standardsFile;
    private $outputFile;
    private $packageRoot;

    public function __construct()
    {
        parent::__construct();
        // Use the actual project root where the command is being executed
        $this->projectRoot = getcwd();
        // Find the package root and read standards from there
        $this->packageRoot = $this->findPackageRoot();
        $this->standardsFile = $this->packageRoot . '/standards/AGENTS.md';
        $this->outputFile = $this->projectRoot . '/AGENTS-with-community.md';
    }

    protected function configure()
    {
        $this->setHelp('This command scans for community-contributed guidelines in .ai/guidelines/* directories and merges them with the main standards.')
             ->addOption('output', 'o', InputOption::VALUE_OPTIONAL, 'Output file path', $this->outputFile)
             ->addOption('dry-run', null, InputOption::VALUE_NONE, 'Show what would be merged without writing files');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $outputPath = $input->getOption('output') ?: $this->outputFile;
        $dryRun = $input->getOption('dry-run');

        $io->title('🔍 Drupal Surge Community Guidelines Scanner');

        // Load main standards
        if (!file_exists($this->standardsFile)) {
            $io->error('Main standards file not found: ' . $this->standardsFile);
            return Command::FAILURE;
        }

        $mainStandards = file_get_contents($this->standardsFile);
        if ($mainStandards === false) {
            $io->error('Could not read main standards file!');
            return Command::FAILURE;
        }

        $io->success('Main standards loaded successfully.');

        // Scan for community guidelines
        $communityGuidelines = $this->scanCommunityGuidelines($io);
        
        if (empty($communityGuidelines)) {
            $io->info('No community guidelines found.');
            if (!$dryRun) {
                $this->writeOutputFile($outputPath, $mainStandards, $io);
            }
            return Command::SUCCESS;
        }

        // Merge guidelines
        $mergedContent = $this->mergeGuidelines($mainStandards, $communityGuidelines, $io);

        if ($dryRun) {
            $io->info('Dry run mode - no files written.');
            $io->writeln("\nMerged content preview:");
            $io->writeln(str_repeat('-', 80));
            $io->writeln(substr($mergedContent, 0, 500) . '...');
            $io->writeln(str_repeat('-', 80));
        } else {
            $this->writeOutputFile($outputPath, $mergedContent, $io);
        }

        return Command::SUCCESS;
    }

    private function scanCommunityGuidelines(SymfonyStyle $io): array
    {
        $guidelines = [];
        $scannedPaths = [];

        // Scan main surge-guidelines directory
        $mainGuidelinesPath = $this->projectRoot . '/surge-guidelines';
        $mainGuidelines = $this->scanDirectory($mainGuidelinesPath);
        if (!empty($mainGuidelines)) {
            $guidelines['Main Guidelines'] = $mainGuidelines;
            $scannedPaths[] = $mainGuidelinesPath;
        }

        // Scan project subdirectories for guidelines recursively
        $projectSubdirs = $this->getProjectSubdirectories();
        
        foreach ($projectSubdirs as $subdir) {
            $this->scanSubdirectoryRecursively($this->projectRoot . '/' . $subdir, $subdir, $guidelines, $scannedPaths, $io);
        }

        $io->info('Scanned paths: ' . implode(', ', $scannedPaths ?: ['none']));
        
        return $guidelines;
    }

    private function getProjectSubdirectories(): array
    {
        $subdirs = [];
        $excludedDirs = [
            'vendor', '.git', 'node_modules', '.idea'
        ];
        
        $items = scandir($this->projectRoot);
        if ($items === false) {
            return $subdirs;
        }
        
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }
            
            $fullPath = $this->projectRoot . '/' . $item;
            if (is_dir($fullPath) && !in_array($item, $excludedDirs)) {
                $subdirs[] = $item;
            }
        }
        
        return $subdirs;
    }

    private function scanSubdirectoryRecursively(string $basePath, string $relativePath, array &$guidelines, array &$scannedPaths, SymfonyStyle $io): void
    {
        // Check for surge-guidelines in current directory
        $guidelinesPath = $basePath . '/surge-guidelines';
        if (is_dir($guidelinesPath)) {
            $subdirGuidelines = $this->scanDirectory($guidelinesPath);
            
            if (!empty($subdirGuidelines)) {
                $guidelines[$relativePath] = $subdirGuidelines;
                $scannedPaths[] = $guidelinesPath;
            }
        }
        
        // Recursively scan subdirectories
        $items = scandir($basePath);
        if ($items === false) {
            return;
        }
        
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }
            
            $fullPath = $basePath . '/' . $item;
            if (is_dir($fullPath)) {
                $newRelativePath = $relativePath . '/' . $item;
                $this->scanSubdirectoryRecursively($fullPath, $newRelativePath, $guidelines, $scannedPaths, $io);
            }
        }
    }

    private function scanDirectory(string $path): array
    {
        $guidelines = [];
        
        if (!is_dir($path)) {
            return $guidelines;
        }

        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile() && $file->getExtension() === 'md') {
                $relativePath = str_replace($path . '/', '', $file->getPathname());
                $content = file_get_contents($file->getPathname());
                
                if ($content !== false) {
                    $guidelines[$relativePath] = $content;
                }
            }
        }

        return $guidelines;
    }

    private function mergeGuidelines(string $mainStandards, array $communityGuidelines, SymfonyStyle $io): string
    {
        $merged = $mainStandards . "\n\n";
        $merged .= "## Community-Contributed Guidelines\n\n";
        $merged .= "The following guidelines have been automatically detected from community-contributed packages and projects.\n\n";

        foreach ($communityGuidelines as $source => $guidelines) {
            $merged .= "### From: {$source}\n\n";
            
            foreach ($guidelines as $filename => $content) {
                $merged .= "#### {$filename}\n\n";
                $merged .= $content . "\n\n";
                $merged .= "---\n\n";
            }
        }

        $io->success('Guidelines merged successfully.');
        
        return $merged;
    }

    private function writeOutputFile(string $outputPath, string $content, SymfonyStyle $io): void
    {
        // Only create directory if the output path includes a subdirectory
        $outputDir = dirname($outputPath);
        if ($outputDir !== '.' && $outputDir !== $this->projectRoot && !is_dir($outputDir)) {
            mkdir($outputDir, 0755, true);
        }

        if (file_put_contents($outputPath, $content) === false) {
            $io->error('Failed to write output file: ' . $outputPath);
            return;
        }

        $io->success('Merged guidelines written to: ' . $outputPath);
    }

    /**
     * Find the package root directory where drupal/surge is installed.
     */
    private function findPackageRoot(): string
    {
        $currentDir = $this->projectRoot;
        while ($currentDir !== '/') {
            if (is_dir($currentDir . '/vendor/drupal/surge')) {
                return $currentDir . '/vendor/drupal/surge';
            }
            $currentDir = dirname($currentDir);
        }
        return $this->projectRoot; // Fallback to project root if package not found
    }

}
