<?php

namespace Drupal\Surge\Console\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;


class StandardsCommand extends Command
{
    protected static $defaultName = 'standards';
    protected static $defaultDescription = 'Apply Drupal Surge coding standards to your preferred IDE/Tool (supports both modern AGENTS.md and IDE-specific formats)';

    private $projectRoot;
    private $standardsFile;
    private $mergedStandardsFile;
    private $packageRoot;

    public function __construct()
    {
        parent::__construct();
        // Use the actual project root where the command is being executed
        $this->projectRoot = getcwd();
        
        // Find the package root and read standards from there
        $this->packageRoot = $this->findPackageRoot();
        $this->standardsFile = $this->packageRoot . '/standards/AGENTS.md';
        $this->mergedStandardsFile = $this->packageRoot . '/standards/AGENTS-with-community.md';
    }

    protected function configure()
    {
        $this->setHelp('This command creates guidelines based on the AGENTS.md file.

Modern AI Agents (AGENTS.md format):
- Compatible with: Cursor, Claude, Aider, Gemini CLI, and 15+ other agents
- Creates a single AGENTS.md file that works across the ecosystem
- Follows the emerging AGENTS.md standard (https://agents.md/)

IDE-Specific Support:
- Creates IDE-specific files for tools that don\'t yet support AGENTS.md
- Includes: Junie (PhpStorm), and other traditional integrations');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        // This command requires interactive mode
        if (!$input->isInteractive()) {
            $io->error('This command requires interactive mode. Please run without --no-interaction flag.');
            $io->error('Usage: php surge standards');
            return Command::FAILURE;
        }

        $io->title('🚀 Drupal Surge Standards Application');

        // Check if merged standards exist, otherwise use base standards
        $standardsFile = $this->mergedStandardsFile;
        $standardsLabel = 'merged standards (with community guidelines)';
        
        if (!file_exists($standardsFile)) {
            $standardsFile = $this->standardsFile;
            $standardsLabel = 'base standards';
            
            if (!file_exists($standardsFile)) {
                $io->error('No standards files found!');
                $io->error('Run "composer surge-scan" first to generate merged standards.');
                return Command::FAILURE;
            }
        }

        $standards = file_get_contents($standardsFile);
        if ($standards === false) {
            $io->error('Could not read standards file!');
            return Command::FAILURE;
        }

        $io->success("Standards file loaded successfully: {$standardsLabel}");
        
        // If using merged standards, show which packages contributed guidelines
        if ($standardsFile === $this->mergedStandardsFile) {
            $this->showContributingPackages($standards, $io);
        }

        $choices = [
            '1' => 'AGENTS.md (Modern - Cursor, Claude, Aider, Gemini CLI, +15 others)',
            '2' => 'IDE-Specific Support',
            '3' => 'Exit'
        ];

        $io->writeln('');
        $io->writeln('Available IDE/Tools:');
        foreach ($choices as $key => $label) {
            $io->writeln("  {$key}. {$label}");
        }
        $io->writeln('');
        
        $choice = $io->ask('Select an option (1-3): ');

        if ($choice === '3') {
            $io->info('👋 Goodbye!');
            return Command::SUCCESS;
        }

        $this->createGuidelines($choice, $standards, $io);

        return Command::SUCCESS;
    }

    private function createGuidelines(string $choice, string $standards, SymfonyStyle $io): void
    {
        switch ($choice) {
            case '1':
                $this->createAgentsMdGuidelines($standards, $io);
                break;
            case '2':
                $this->showSpecificOptions($standards, $io);
                break;
        }
    }

    private function createAgentsMdGuidelines(string $standards, SymfonyStyle $io): void
    {
        $targetFile = $this->projectRoot . '/AGENTS.md';

        // Determine which source file to use (merged or base)
        $sourceFile = $this->mergedStandardsFile;
        if (!file_exists($sourceFile)) {
            $sourceFile = $this->standardsFile;
        }

        // Copy the existing AGENTS.md file directly
        if (!copy($sourceFile, $targetFile)) {
            $io->error('Failed to copy AGENTS.md file to project root');
            return;
        }

        $io->success('Created AGENTS.md file in project root');
        $io->writeln('');
        $io->writeln('<info>✅ Compatible with modern AI agents:</info>');
        $io->writeln('  • Cursor, Claude, Aider, Gemini CLI');
        $io->writeln('  • Codex, Amp, Jules, Factory, RooCode');
        $io->writeln('  • Kilo Code, opencode, Phoenix, Zed, Semgrep, Warp');
        $io->writeln('');
        $io->writeln('<comment>📖 Learn more: https://agents.md/</comment>');
        $io->writeln('');
        $io->writeln('<comment>💡 Tip: Use "IDE-Specific Support" to configure Aider and Gemini CLI</comment>');
    }

    private function showSpecificOptions(string $standards, SymfonyStyle $io): void
    {
        $io->writeln('');
        $io->writeln('<info>IDE-Specific Support Options:</info>');
        $io->writeln('');
        
        $choices = [
            '1' => 'Junie (PhpStorm)',
            '2' => 'Claude Code (Claude.md format)',
            '3' => 'Aider (.aider.conf.yml), configures Agents.md',
            '4' => 'Gemini CLI (.gemini/settings.json), configures Agents.md',
            '5' => 'Custom (Copy-paste to your tool)',
            '6' => 'Back to Main Menu'
        ];

        foreach ($choices as $key => $label) {
            $io->writeln("  {$key}. {$label}");
        }
        $io->writeln('');
        
        $choice = $io->ask('Select an IDE-specific option (1-6): ');

        switch ($choice) {
            case '1':
                $this->createJunieGuidelines($standards, $io);
                break;
            case '2':
                $this->createClaudeGuidelines($standards, $io);
                break;
            case '3':
                $this->createAiderConfig($io);
                break;
            case '4':
                $this->createGeminiConfig($io);
                break;
            case '5':
                $this->createCustomGuidelines($standards, $io);
                break;
            case '6':
                $io->info('Returning to main menu...');
                // Note: In a real implementation, you'd need to restart the command
                // For now, we'll just show a message
                $io->writeln('Please run the command again to return to the main menu.');
                break;
        }
    }

    private function createCustomGuidelines(string $standards, SymfonyStyle $io): void
    {
        // Determine which source file to use (merged or base)
        $sourceFile = $this->mergedStandardsFile;
        if (!file_exists($sourceFile)) {
            $sourceFile = $this->standardsFile;
        }

        // Read the content
        $content = file_get_contents($sourceFile);
        if ($content === false) {
            $io->error('Failed to read guidelines content');
            return;
        }

        $io->success('📋 Drupal Surge Guidelines - Copy & Paste Ready');
        $io->writeln('');
        $io->writeln('<info>Copy the content below and paste it into your preferred AI coding tool or IDE:</info>');
        $io->writeln('');
        $io->writeln(str_repeat('=', 80));
        $io->writeln('');
        $io->writeln($content);
        $io->writeln('');
        $io->writeln(str_repeat('=', 80));
        $io->writeln('');
        $io->writeln('<comment>💡 Tip: Select all the content above and copy it to your clipboard</comment>');
        $io->writeln('<comment>   Then paste it into your AI coding tool or IDE configuration.</comment>');
    }

    private function createJunieGuidelines(string $standards, SymfonyStyle $io): void
    {
        $targetDir = $this->projectRoot . '/.junie';
        $targetFile = $targetDir . '/guidelines.md';

        $this->ensureDirectory($targetDir);

        $content = $this->formatForJunie($standards);
        file_put_contents($targetFile, $content);

        $io->success('Created Junie guidelines at: .junie/guidelines.md');
    }



    private function createClaudeGuidelines(string $standards, SymfonyStyle $io): void
    {
        $targetFile = $this->projectRoot . '/Claude.md';

        $content = $this->formatForClaude($standards);
        file_put_contents($targetFile, $content);

        $io->success('Created Claude guidelines at: Claude.md');
    }

    private function formatForJunie(string $standards): string
    {
        return "# Drupal Surge - Junie (PhpStorm) Guidelines\n\n" .
               "These guidelines are based on the Drupal Surge standards and optimized for PhpStorm development.\n\n" .
               "---\n\n" .
               $standards;
    }



    private function formatForClaude(string $standards): string
    {
        return "# Drupal Surge - Claude Code Guidelines\n\n" .
               "These guidelines are based on the Drupal Surge standards and optimized for Claude Code development.\n\n" .
               "---\n\n" .
               $standards;
    }



    private function createAiderConfig(SymfonyStyle $io): void
    {
        $configFile = $this->projectRoot . '/.aider.conf.yml';
        
        // Check if config already exists
        if (file_exists($configFile)) {
            $io->writeln('<comment>⚠️  .aider.conf.yml already exists - skipping Aider configuration</comment>');
            return;
        }

        $config = "read: AGENTS.md\n";
        
        if (file_put_contents($configFile, $config) === false) {
            $io->writeln('<comment>⚠️  Could not create .aider.conf.yml</comment>');
            return;
        }

        $io->writeln('<info>✅ Created .aider.conf.yml for Aider configuration</info>');
    }

    private function createGeminiConfig(SymfonyStyle $io): void
    {
        $geminiDir = $this->projectRoot . '/.gemini';
        $configFile = $geminiDir . '/settings.json';
        
        // Check if config already exists
        if (file_exists($configFile)) {
            $io->writeln('<comment>⚠️  .gemini/settings.json already exists - skipping Gemini CLI configuration</comment>');
            return;
        }

        // Create .gemini directory if it doesn't exist
        if (!is_dir($geminiDir)) {
            if (!mkdir($geminiDir, 0755, true)) {
                $io->writeln('<comment>⚠️  Could not create .gemini directory</comment>');
                return;
            }
        }

        $config = json_encode(['contextFileName' => 'AGENTS.md'], JSON_PRETTY_PRINT);
        
        if (file_put_contents($configFile, $config) === false) {
            $io->writeln('<comment>⚠️  Could not create .gemini/settings.json</comment>');
            return;
        }

        $io->writeln('<info>✅ Created .gemini/settings.json for Gemini CLI configuration</info>');
    }

    private function showContributingPackages(string $standards, SymfonyStyle $io): void
    {
        $io->writeln('');
        $io->writeln('📚 <info>Community Guidelines Contributors:</info>');
        
        // Extract package names from the merged standards
        if (preg_match_all('/### From: (.+)/', $standards, $matches)) {
            foreach ($matches[1] as $package) {
                $io->writeln("  • <comment>{$package}</comment>");
            }
        } else {
            $io->writeln('  • <comment>No community guidelines found</comment>');
        }
        
        $io->writeln('');
    }

    private function ensureDirectory(string $dir): void
    {
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
    }

    private function findPackageRoot(): string
    {
        $currentDir = $this->projectRoot;
        while ($currentDir !== '/') {
            if (is_dir($currentDir . '/vendor/drupal/surge')) {
                return $currentDir . '/vendor/drupal/surge';
            }
            $currentDir = dirname($currentDir);
        }
        return $this->projectRoot; // Fallback to project root if package not found
    }
}
