<?php

namespace Drupal\Surge\Composer;

use Composer\Composer;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\IO\IOInterface;
use Composer\Plugin\PluginInterface;
use Composer\Script\Event;
use Composer\Json\JsonFile;

/**
 * Composer plugin to auto-register Surge scripts in consuming projects.
 */
class SurgePlugin implements PluginInterface, EventSubscriberInterface
{
    /**
     * {@inheritdoc}
     */
    public function activate(Composer $composer, IOInterface $io)
    {
        // Plugin activation logic
    }

    /**
     * {@inheritdoc}
     */
    public function deactivate(Composer $composer, IOInterface $io)
    {
        // Plugin deactivation logic
    }

    /**
     * {@inheritdoc}
     */
    public function uninstall(Composer $composer, IOInterface $io)
    {
        // Plugin uninstall logic
    }

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents()
    {
        return [
            'post-install-cmd' => 'registerScripts',
            'post-update-cmd' => 'registerScripts',
        ];
    }

    /**
     * Register Surge scripts in the consuming project's composer.json.
     */
    public function registerScripts(Event $event)
    {
        $composer = $event->getComposer();
        $io = $event->getIO();
        
        // Get the project root directory where composer was executed
        $projectRoot = getcwd();
        $composerJsonPath = $projectRoot . '/composer.json';
        
        // Check if this is the root project (not a dependency)
        if (!file_exists($composerJsonPath)) {
            return;
        }
        
        // Don't run if we're in the surge package directory itself
        $surgePackageDir = dirname(__DIR__, 3);
        if ($projectRoot === $surgePackageDir) {
            return;
        }
        
        try {
            $jsonFile = new JsonFile($composerJsonPath);
            $jsonContent = $jsonFile->read();
            
            // Check if scripts section exists
            if (!isset($jsonContent['scripts'])) {
                $jsonContent['scripts'] = [];
            }
            
            $scriptsAdded = false;
            
            // Add surge-standards script if it doesn't exist
            if (!isset($jsonContent['scripts']['surge-standards'])) {
                $jsonContent['scripts']['surge-standards'] = 'Drupal\\Surge\\Console\\SurgeApplication::runStandards';
                $scriptsAdded = true;
            }
            
            // Add surge-scan script if it doesn't exist
            if (!isset($jsonContent['scripts']['surge-scan'])) {
                $jsonContent['scripts']['surge-scan'] = 'Drupal\\Surge\\Console\\SurgeApplication::runScan';
                $scriptsAdded = true;
            }
            
            // Only write if we added scripts
            if ($scriptsAdded) {
                $jsonFile->write($jsonContent);
                $io->write('<info>Surge scripts have been automatically registered in your composer.json</info>');
                $io->write('<info>You can now use: composer surge-standards and composer surge-scan</info>');
            }
            
        } catch (\Exception $e) {
            $io->writeError('<warning>Could not auto-register Surge scripts: ' . $e->getMessage() . '</warning>');
            $io->writeError('<warning>Please manually add the scripts to your composer.json</warning>');
        }
    }
}
