/* eslint-disable import/no-unresolved */
import { Plugin } from 'ckeditor5/src/core';
import { toWidget, Widget } from 'ckeditor5/src/widget';
import InsertSvgSpriteCommand from './svg_sprite-insertcommand';

/**
 * Returns a converter that consumes the sprite svg element.
 *
 * @return {Function}
 *   A function that adds an event listener to upcastDispatcher.
 */
function upcastSvgSprite() {
  return (dispatcher) => {
    dispatcher.on(
      'element:span',
      (evt, data, conversionApi) => {
        // Get all the necessary items from the conversion API object.
        const { consumable, writer, safeInsert, updateConversionResult } =
          conversionApi;

        // Get view item from the data object.
        const { viewItem: spanItem } = data;

        // Check if there is only one child.
        if (spanItem.childCount !== 1) {
          return;
        }

        // Get the first child element.
        const svgItem = spanItem.getChild(0);

        // Check if the first element is a <svg>.
        if (!svgItem.is('element', 'svg')) {
          return;
        }

        // Check if there is only one child.
        if (svgItem.childCount !== 1) {
          return;
        }

        // Get the first child element.
        const useItem = svgItem.getChild(0);

        // Check if the first element is a <use>.
        if (!useItem.is('element', 'use')) {
          return;
        }

        // Check if the view element has already been processed.
        if (consumable.consume(spanItem, { name: true })) {
          // Create the model element.
          const modelElement = writer.createElement('svgSprite');

          const href = useItem.getAttribute('href');

          // Extract the sprite ID from the href.
          const spriteIdMatch = href.match(/#(.*)$/);
          if (spriteIdMatch) {
            const spriteId = spriteIdMatch[1];
            writer.setAttribute('sprite_id', spriteId, modelElement);
          } else {
            return;
          }

          // Insert element on a current cursor location.
          if (!safeInsert(modelElement, data.modelCursor)) {
            return;
          }

          // Necessary function call to help setting the model range and cursor
          // for some specific cases when elements being split.
          updateConversionResult(modelElement, data);

          // Mark the event as handled, so it won't be upcasted again.
          evt.stop();
        }
      },
      { priority: 'high' },
    );
  };
}

/**
 * Builds the <i class="icon"><svg><use> component using an SVG sprite.
 *
 * @private
 */
class SvgSpriteEditing extends Plugin {
  /**
   * @inheritdoc
   */
  static get requires() {
    return [Widget];
  }

  /**
   * @inheritdoc
   */
  static get pluginName() {
    return 'SvgSpriteEditing';
  }

  /**
   * @inheritdoc
   */
  init() {
    const { editor } = this;
    const { schema } = editor.model;

    const config = editor.config.get('svgSprite');

    editor.commands.add('insertSvgSprite', new InsertSvgSpriteCommand(editor));

    //
    // SvgSprite
    //
    schema.register('svgSprite', {
      inheritAllFrom: '$inlineObject',
      allowAttributes: ['sprite_id'],
    });

    editor.conversion.for('upcast').add(upcastSvgSprite());

    editor.conversion.for('dataDowncast').elementToElement({
      model: 'svgSprite',
      view: (modelElement, { writer: viewWriter }) => {
        const spriteId = modelElement.getAttribute('sprite_id');
        const { href } = config;

        // Create span element with class.
        const spanElement = viewWriter.createContainerElement('span', {
          class: 'svg-sprite',
        });

        // Create svg element with classes and attributes.
        const svgElement = viewWriter.createContainerElement('svg', {
          class: `sprite sprite-${spriteId}`,
          'aria-hidden': 'true',
          focusable: 'false',
        });

        // Create use element with href.
        const useElement = viewWriter.createEmptyElement('use', {
          href: `${href}#${spriteId}`,
        });

        // Build the structure: span > svg > use.
        viewWriter.insert(
          viewWriter.createPositionAt(svgElement, 0),
          useElement,
        );
        viewWriter.insert(
          viewWriter.createPositionAt(spanElement, 0),
          svgElement,
        );

        return spanElement;
      },
    });

    editor.conversion.for('editingDowncast').elementToElement({
      model: 'svgSprite',
      view: (modelElement, { writer: viewWriter }) => {
        const spanElement = viewWriter.createContainerElement('span', null);

        const rawElement = viewWriter.createRawElement(
          'span',
          { class: 'svg-sprite' },
          (domElement) => {
            const { href } = config;
            const spriteId = modelElement.getAttribute('sprite_id');
            domElement.innerHTML = `<svg class="sprite sprite-${spriteId}" aria-hidden="true" focusable="false"><use href="${href}#${spriteId}"></use></svg>`;
          },
        );

        viewWriter.insert(
          viewWriter.createPositionAt(spanElement, 0),
          rawElement,
        );

        return toWidget(spanElement, viewWriter);
      },
    });
  }
}

export default SvgSpriteEditing;
