<?php

namespace Drupal\svg_sprite\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'svg_sprite' widget.
 */
#[FieldWidget(
  id: 'svg_sprite',
  label: new TranslatableMarkup('SVG Sprite'),
  field_types: ['svg_sprite'],
)]
class SvgSpriteWidget extends WidgetBase {

  /**
   * The svg_sprite module configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * Abstract over the actual field columns.
   *
   * Allows different field types to reuse those widgets.
   *
   * @var string
   */
  protected $column;

  /**
   * Tracks whether the field is required.
   */
  protected bool $required;

  /**
   * Tracks whether the field has a value.
   */
  protected bool $hasValue;

  /**
   * The array of options for the widget.
   */
  protected array $options;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $property_names = $instance->fieldDefinition->getFieldStorageDefinition()->getPropertyNames();
    $instance->column = $property_names[0];
    $instance->config = $container->get('config.factory')->get('svg_sprite.settings');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {

    $this->required = $element['#required'];
    $this->hasValue = isset($items[$delta]->{$this->column});

    $element['value'] = $element + [
      '#type' => 'select',
      '#options' => $this->getOptions($items->getEntity()),
      '#default_value' => $this->getSelectedOption($items, $delta),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  protected function getOptions(FieldableEntityInterface $entity) {
    if (!isset($this->options)) {

      $options = $this->config->get('symbols') ?? [];

      // Add an empty option if the widget needs one.
      if ($empty_label = $this->getEmptyLabel()) {
        $options = ['' => $empty_label] + $options;
      }

      array_walk_recursive($options, [$this, 'sanitizeLabel']);

      $this->options = $options;
    }
    return $this->options;
  }

  /**
   * Determines selected options from the incoming field values.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The field values.
   * @param int $delta
   *   The index of the field value.
   *
   * @return array
   *   The array of corresponding selected options.
   */
  protected function getSelectedOption(FieldItemListInterface $items, $delta) {
    $selected_option = NULL;
    $value = $items[$delta]->{$this->column};
    if (isset($this->options[$value])) {
      $selected_option = $value;
    }
    return $selected_option;
  }

  /**
   * {@inheritdoc}
   */
  protected function sanitizeLabel(&$label) {
    // Select form inputs allow unencoded HTML entities, but no HTML tags.
    $label = Html::decodeEntities(strip_tags($label));
  }

  /**
   * {@inheritdoc}
   */
  protected function getEmptyLabel() {
    // Single select: add a 'none' option for non-required fields,
    // and a 'select a value' option for required fields that do not come
    // with a value selected.
    if (!$this->required) {
      return $this->t('- None -');
    }
    if (!$this->hasValue) {
      return $this->t('- Select a value -');
    }
  }

}
