<?php

namespace Drupal\svg_sprite\Form;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\svg_sprite\Service\SvgSpriteService;
use Drupal\svg_sprite\SvgSpriteHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class SvgSpriteSettingsForm.
 *
 * The settings form of the svg_sprite module.
 *
 * @package Drupal\svg_sprite\Form
 */
class SvgSpriteSettingsForm extends SettingsFormBase {

  const SETTINGS = 'svg_sprite.settings';

  /**
   * The SVG Sprite service.
   *
   * @var \Drupal\svg_sprite\Service\SvgSpriteService
   */
  protected SvgSpriteService $svgSpriteService;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->svgSpriteService = $container->get('svg_sprite');
    $instance->renderer = $container->get('renderer');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'svg_sprite_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getSettings() {
    return self::SETTINGS;
  }

  /**
   * {@inheritdoc}
   */
  protected function addFormElements(array &$form, $config) {

    $form['source'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Source of the SVG sprites file'),
      '#description' => $this->t("You can provide a full URL or specify a path relative to Drupal's web directory.<br/>You can also use the theme://[theme_name]/ prefix to target an SVG file in a theme."),
      '#default_value' => $config->get('source'),
      '#required' => TRUE,
    ];

    $form['alpha_sort'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Sort sprites alphabetically'),
      '#description' => $this->t("Will sort sprites alphabetically using the symbol's <code>id</code>, or <code>aria-label</code>, if provided.<br/>Leave unchecked to keep the ordering of the SVG source file."),
      '#default_value' => $config->get('alpha_sort'),
      '#required' => FALSE,
    ];

    $source = $config->get('source');
    if ($source) {
      $data = $this->svgSpriteService->fetchSvgDataFromSource($source);

      if ($data) {
        $alpha_sort = $config->get('alpha_sort');
        $symbols = $this->svgSpriteService->extractSpriteInfoFromSvgData($data, $alpha_sort);

        $svg_sprites_html = '';
        foreach ($symbols as $id => $symbol) {
          $svg_sprites_html .= '<div>';
          $render_array = SvgSpriteHelper::buildSvgSpriteRenderArray($id);
          $svg_sprites_html .= $this->renderer->renderInIsolation($render_array);
          $svg_sprites_html .= ' ' . $id;
          if ($symbol !== $id) {
            $svg_sprites_html .= ' (' . $symbol . ')';
          }
          $svg_sprites_html .= '</div>';
        }

        $svg_sprites_title = $this->t('Sprite preview');

        $form['preview'] = [
          '#type' => 'markup',
          '#markup' => '<h3>' . $svg_sprites_title . '</h3><div class="svg-sprite-preview"><div>' . $svg_sprites_html . '</div></div>',
          '#allowed_tags' => array_merge(Xss::getAdminTagList(), ['svg', 'use']),
        ];

        $form['#attached']['library'][] = 'svg_sprite/settings';
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function validateFormElements(FormStateInterface $form_state, $config) {
    $source = $form_state->getValue('source');
    $data = $this->svgSpriteService->fetchSvgDataFromSource($source);
    if (!$data) {
      $form_state->setErrorByName('source', $this->t('Invalid SVG source.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function saveFormElements(FormStateInterface $form_state, $config) {
    $source = $form_state->getValue('source');
    $alpha_sort = $form_state->getValue('alpha_sort');
    $data = $this->svgSpriteService->fetchSvgDataFromSource($source);
    $symbols = $this->svgSpriteService->extractSpriteInfoFromSvgData($data, $alpha_sort);
    $href = $this->svgSpriteService->getSvgHref($source);
    $config
      ->set('source', $source)
      ->set('alpha_sort', $alpha_sort)
      ->set('symbols', $symbols)
      ->set('href', $href)
      ->save();
  }

}
