<?php

namespace Drupal\svg_sprite\Service;

use Drupal\Core\Extension\ThemeExtensionList;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Error;
use GuzzleHttp\Client;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Various helper functions for the svg_sprite module.
 *
 * @package Drupal\svg_sprite
 */
class SvgSpriteService {

  use LoggerChannelTrait;
  use StringTranslationTrait;

  const SOURCE_THEME_PATTERN = '/theme:\/\/([^\/]+)\/(.*)/';

  /**
   * The theme manager.
   *
   * @var \Drupal\Core\Extension\ThemeExtensionList
   */
  protected ThemeExtensionList $extensionListTheme;

  /**
   * HTTP client service.
   *
   * @var \GuzzleHttp\Client
   */
  protected Client $httpClient;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected Request $request;

  /**
   * Constructor of the SVG Sprite service.
   *
   * @param \Drupal\Core\Extension\ThemeExtensionList $extension_list_theme
   *   The theme manager.
   * @param \GuzzleHttp\Client $httpClient
   *   The Drupal http client.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(ThemeExtensionList $extension_list_theme, Client $httpClient, RequestStack $request_stack) {
    $this->extensionListTheme = $extension_list_theme;
    $this->httpClient = $httpClient;
    $this->request = $request_stack->getCurrentRequest();
  }

  /**
   * Fetch SVG data from the specified source.
   */
  public function fetchSvgDataFromSource(string $svg_source) {
    if (str_starts_with($svg_source, 'http')) {
      try {
        $response = $this->httpClient->get($svg_source);
        return $response->getBody()->getContents();
      }
      catch (\Throwable $e) {
        $this->getLogger('svg_sprite')->error($e->getMessage());
      }
    }
    else {
      if (str_starts_with($svg_source, 'theme://')) {
        if (preg_match(self::SOURCE_THEME_PATTERN, $svg_source, $matches)) {
          $theme_name = $matches[1];
          $asset_path = $matches[2];
          $theme_path = $this->extensionListTheme->getPath($theme_name);
          $svg_file = $theme_path . '/' . $asset_path;
        }
        else {
          return FALSE;
        }
      }
      else {
        $svg_file = trim($svg_source, '/');
      }
      return file_get_contents($svg_file);
    }
    return FALSE;
  }

  /**
   * Fetch SVG data from the specified source.
   */
  public function getSvgHref(string $svg_source) {
    if (str_starts_with($svg_source, 'http')) {
      return $svg_source;
    }
    else {
      if (str_starts_with($svg_source, 'theme://')) {
        if (preg_match(self::SOURCE_THEME_PATTERN, $svg_source, $matches)) {
          $theme_name = $matches[1];
          $asset_path = $matches[2];
          $theme_path = $this->extensionListTheme->getPath($theme_name);
          $svg_file = $theme_path . '/' . $asset_path;
        }
        else {
          return FALSE;
        }
      }
      else {
        $svg_file = trim($svg_source, '/');
      }
      $href = $this->request->getBasePath() . '/' . $svg_file;
      if (file_exists($svg_file)) {
        $href .= '?v=' . filemtime($svg_file);
      }
      return $href;
    }
  }

  /**
   * Extracts IDs and Aria labels from an SVG sprites file.
   */
  public function extractSpriteInfoFromSvgData(string $svg_data, bool $alpha_sort): array {
    $sprites = [];

    try {
      $xml = new \SimpleXMLElement($svg_data);

      $defs = $xml->defs;

      if ($defs) {

        $symbols = $defs->symbol;

        foreach ($symbols as $symbol) {
          $id = (string) $symbol['id'];
          $label = empty($symbol['aria-label']) ? $id : (string) $symbol['aria-label'];
          // phpcs:ignore Drupal.Semantics.FunctionT.NotLiteralString
          $sprites[$id] = (string) $this->t($label);
        }

        if ($alpha_sort) {
          asort($sprites);
        }
      }
    }
    catch (\Throwable $e) {
      Error::logException($this->getLogger('svg_sprite'), $e);
    }

    return $sprites;
  }

}
