<?php

namespace Drupal\svg_sprite;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Template\Attribute;

/**
 * Various helper functions for the svg_sprite module.
 */
class SvgSpriteHelper {

  /**
   * List of allowed admin HTML tags for SVG Sprite token compatibility.
   *
   * @var array
   */
  protected static array $xssAdminTags;

  /**
   * Add 'svg' and 'use' to the allowed admin tags.
   *
   * This allows usage of the svg_sprite token in some restricted contexts.
   */
  public static function addSvgSpriteTagsToAdminTags(): void {
    if (!isset(self::$xssAdminTags)) {
      $reflection = new \ReflectionClass(Xss::class);
      $property = $reflection->getProperty('adminTags');
      $property->setAccessible(TRUE);
      $admin_tags = $property->getValue(NULL);
      $admin_tags = array_merge($admin_tags, ['svg', 'use']);
      $admin_tags = array_unique($admin_tags);
      $property->setValue(NULL, $admin_tags);
      self::$xssAdminTags = $admin_tags;
    }
  }

  /**
   * Builds an SVG sprite render array.
   *
   * @param string $sprite_id
   *   The sprite id.
   * @param string $custom_attributes
   *   The custom attributes.
   *
   * @return array
   *   The render array.
   */
  public static function buildSvgSpriteRenderArray(string $sprite_id, ?Attribute $custom_attributes = NULL): array {
    $attributes = new Attribute([
      'class' => ['sprite', 'sprite-' . $sprite_id],
      'aria-hidden' => 'true',
      'focusable' => 'false',
    ]);
    if ($custom_attributes) {
      $attributes->merge($custom_attributes);
    }
    return [
      '#theme' => 'svg_sprite',
      '#attributes' => $attributes,
      '#sprite_id' => $sprite_id,
    ];
  }

}
