# SVG.js

## INTRODUCTION

The SVG.js module integrates the powerful SVG.js library with Drupal, providing
a lightweight and efficient way to manipulate and animate SVG elements on your
Drupal site.

SVG.js is a lightweight library for manipulating and animating SVG. It provides
an easy-to-use API for creating, manipulating and animating SVG elements with
no dependencies and a small file size.

## REQUIREMENTS

This module requires the following:

- Drupal 8.8 or higher (including Drupal 9, 10, and 11)
- SVG.js library (automatically loaded via CDN or can be installed locally)

## INSTALLATION

### Method 1: Using Composer (Recommended)

```bash
  composer require drupal/svgjs
  drush en svgjs
```

### Method 2: Manual Installation

Download the module and place it in your modules directory
(e.g., /modules/contrib/svgjs)

**Enable the module**:

- **Via Drush**:
```bash
  drush en svgjs
```
- **Via UI**: Navigate to Extend page and enable "SVG.js"

### Library Installation (Optional)
By default, the module loads SVG.js from CDN. To use a local copy:

  1. Download SVG.js from https://github.com/svgdotjs/svg.js/releases
  2. Extract the files to /libraries/svgjs/
  3. Ensure the file exists at /libraries/svgjs/dist/svg.min.js

## CONFIGURATION
The module works out of the box with CDN loading. No configuration is required
for basic usage.

### CDN vs Local Loading
- The module automatically detects if the SVG.js library is installed locally.
- If not found, it will load from CDN (cdn.jsdelivr.net).

## USAGE
**Basic Implementation**\
Once enabled, the SVG.js library will be available on your pages. You can use
it in your custom JavaScript:

```javascript
  // Create an SVG drawing
  var draw = SVG().addTo('#drawing').size(300, 300);

  // Draw a rectangle
  var rect = draw.rect(100, 100).fill('#f06');

  // Animate the rectangle
  rect.animate(1000).move(150, 150).rotate(45);
```

**In Your Theme**\
Add SVG.js functionality to your theme's JavaScript:

```javascript
  (function ($, Drupal) {
    Drupal.behaviors.myThemeSvg = {
      attach: function (context, settings) {
        // Your SVG.js code here
        once('my-svg', '.svg-container', context).forEach(function(element) {
          var draw = SVG().addTo(element).size('100%', '100%');
          // Create your SVG elements
        });
      }
    };
  })(jQuery, Drupal);
```

**In Custom Modules**\
You can programmatically attach the SVG.js library:
```php
  function mymodule_page_attachments(array &$attachments) {
    $attachments['#attached']['library'][] = 'svgjs/svgjs.js';
  }
```

## FEATURES

- **Lightweight:** No dependencies, small file size
- **Powerful API**: Intuitive methods for SVG manipulation
- **Animation Support**: Built-in animation capabilities
- **Method Chaining**: Clean, readable code
- **Plugin System**: Extensible architecture
- **Event Handling**: Full event support for interactivity
- **Transform Support**: Rotate, scale, translate, and skew
- **Gradients & Patterns**: Advanced styling options
- **Text Support**: SVG text manipulation
- **Masking & Clipping**: Advanced visual effects

## API EXAMPLES

### Creating Elements

```javascript
  // Rectangle
  var rect = draw.rect(100, 100).fill('#f06');

  // Circle
  var circle = draw.circle(100).fill('#f06').move(50, 50);

  // Ellipse
  var ellipse = draw.ellipse(200, 100).fill('#f06');

  // Line
  var line = draw.line(0, 0, 100, 150).stroke({ width: 1 });

  // Polyline
  var polyline = draw.polyline('0,0 100,50 50,100')
    .fill('none')
    .stroke({ width: 1 });

  // Polygon
  var polygon = draw.polygon('0,0 100,50 50,100').fill('#f06');

  // Path
  var path = draw.path('M0 0 L100 100 Z').fill('none').stroke({ width: 1 });
```

### Animations

```javascript
  // Basic animation
  rect.animate(1000).move(200, 200);

  // Animation with easing
  rect.animate(1000, '<>').move(200, 200);

  // Chained animations
  rect
  .animate(1000).move(200, 200)
  .animate(1000).rotate(180)
  .animate(1000).scale(2);

  // Loop animation
  rect.animate(1000).move(200, 200).loop();
```

### Events

```javascript
  // Click event
  rect.click(function() {
    this.fill({ color: '#f00' });
  });

  // Mouse events
  rect.mouseover(function() {
    this.fill({ color: '#f00' });
  });

  rect.mouseout(function() {
    this.fill({ color: '#f06' });
  });
```

### Transformations

```javascript
  // Move
  rect.move(100, 100);

  // Rotate
  rect.rotate(45);

  // Scale
  rect.scale(2);

  // Skew
  rect.skew(30, 0);

  // Matrix transformation
  rect.matrix(1, 0, 0, 1, 100, 100);
```

## BROWSER SUPPORT
SVG.js supports all modern browsers:

* Chrome
* Firefox
* Safari
* Edge
* Opera

Internet Explorer 11 is supported with some limitations.

## TROUBLESHOOTING

**Library Not Loading**\
If the SVG.js library is not loading:

* Check browser console for errors
* Verify the library path if using local installation
* Clear Drupal cache: drush cr
* Check if the module is properly enabled

**Conflicts with Other Libraries**\
If experiencing conflicts:

* Check loading order of JavaScript libraries
* Use jQuery.noConflict() if needed
* Ensure no duplicate SVG.js library loading

**Performance Issues**\
For better performance:

* Use local library instead of CDN for production sites
* Minimize the number of SVG elements
* Use CSS transforms when possible
* Optimize SVG files before use

## RESOURCES

* SVG.js Documentation: https://svgjs.dev/docs/3.0/
* SVG.js GitHub: https://github.com/svgdotjs/svg.js
* SVG.js Plugins: https://github.com/svgdotjs/svg.js#plugins
* Examples: https://svgjs.dev/docs/3.0/examples/

### Maintainers
Current maintainer:

**Mahyar SBT** (mahyarsbt) -
https://www.drupal.org/u/mahyarsbt

## LICENSE
This module is licensed under the GNU General Public License v2.0 or later.
The SVG.js library is licensed under the MIT License.
