<?php

namespace Drupal\swagger_php\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\swagger_php\OpenApiSpecGenerator;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Controller for Swagger PHP's generated OpenAPI documentation.
 */
class OpenApiController extends ControllerBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('swagger_php.generator')
    );
  }

  /**
   * Constructs a new OpenApiController object.
   *
   * @param \Drupal\swagger_php\OpenApiSpecGenerator $openApiSpecGenerator
   *   The OpenAPI specification generator service.
   */
  public function __construct(private readonly OpenApiSpecGenerator $openApiSpecGenerator) {}

  /**
   * Returns the OpenAPI specification as a JSON response.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response containing the OpenAPI spec.
   */
  public function specJson(): JsonResponse {
    // Get the generated JSON spec from the service.
    $spec_json = $this->openApiSpecGenerator->generateJsonSpec();
    return new JsonResponse(
      json_decode($spec_json),
      200,
      ['Content-Type' => 'application/json']
    );
  }

  /**
   * Renders the Swagger UI documentation page.
   *
   * @return array
   *   A render array for the Swagger UI page.
   */
  public function swaggerUi(): array {
    return [
      '#theme' => 'swagger_php_ui',
      '#attached' => [
        'library' => [
          'swagger_php/swagger-load',
        ],
      ],
    ];
  }

}
