/**
 * @file
 * Contains definition of the behaviour Swup settings.
 */

(function ($, window, Drupal, once) {
  "use strict";

  /**
   * Provide the summary information for the swup settings vertical tabs.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the behavior for the swup settings summaries.
   */
  Drupal.behaviors.swupSettingsSummary = {
    attach() {
      // The drupalSetSummary method required for this behavior is not available
      // on the Swup settings page, so we need to make sure this behavior
      // is processed only if drupalSetSummary is defined.
      if (typeof $.fn.drupalSetSummary === 'undefined') {
        return;
      }

      $(once('swup-files', '[data-drupal-selector="edit-files"]'),
      ).drupalSetSummary((context) => {
        const enabled = {
          core: $(context).find('[data-drupal-selector="edit-core"]').prop('checked'),
          js: $(context).find('[data-drupal-selector="edit-js"]').prop('checked'),
          preload: $(context).find('[data-drupal-selector="edit-preload"]').prop('checked'),
          scripts: $(context).find('[data-drupal-selector="edit-scripts"]').prop('checked'),
          forms: $(context).find('[data-drupal-selector="edit-forms"]').prop('checked'),
          debug: $(context).find('[data-drupal-selector="edit-debug"]').prop('checked'),
        };

        // If core is disabled, nothing works.
        if (!enabled.core) {
          return Drupal.t('Core disabled (Swup inactive)');
        }

        // Check if all plugins are enabled.
        if (enabled.js && enabled.preload && enabled.scripts && enabled.forms && enabled.debug) {
          return Drupal.t('Core and all plugins enabled');
        }

        const values = [Drupal.t('Core enabled')];
        if (enabled.js) {
          values.push(Drupal.t('JS'));
        }
        if (enabled.preload) {
          values.push(Drupal.t('Preload'));
        }
        if (enabled.scripts) {
          values.push(Drupal.t('Scripts'));
        }
        if (enabled.forms) {
          values.push(Drupal.t('Forms'));
        }
        if (enabled.debug) {
          values.push(Drupal.t('Debug'));
        }

        return values.join(', ');
      });

      $(once('swup-theme-groups', '[data-drupal-selector="edit-theme-groups"]'),
      ).drupalSetSummary((context) => {
        const $themes = $(context).find(
          'select[name="themes[]"]',
        );
        if (!$themes.length || !$themes[0].value) {
          return Drupal.t('Not restricted');
        }

        return Drupal.t('Restricted to !theme', { '!theme': $themes.val() });
      });

      $(once('swup-request-path', '[data-drupal-selector="edit-request-path"]'),
      ).drupalSetSummary((context) => {
        const $pages = $(context).find(
          'textarea[name="pages"]',
        );
        if (!$pages.length || !$pages[0].value) {
          return Drupal.t('Not restricted');
        }

        return Drupal.t('Restricted to certain pages');
      });
    },
  };

})(jQuery, window, Drupal, once);
