<?php

declare(strict_types=1);

namespace Drupal\symfony_mailer_addons\Form;

use Drupal\Core\Entity\Element\EntityAutocomplete;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Symfony Mailer Addons form.
 */
class EmailFooterLinksForm extends FormBase {

  /**
   * Constructs a new EmailFooterLinksForm object.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected StateInterface $state,
    protected LanguageManagerInterface $languageManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('state'),
      $container->get('language_manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'symfony_mailer_addons_email_footer_links';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $languages = $this->languageManager->getLanguages();

    $form['description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Add links to be displayed in the footer of emails sent by Symfony Mailer. These links will be available in all languages.<br>Example of usage in the template:<br><pre>@code</pre>', [
        '@code' => <<<TWIG
{% if footer_links is not empty %}
  <tr>
    <table border="0" cellpadding="0" cellspacing="0" width="100%">
      <tr>
        {% for link in footer_links %}
          <td>{{ link }}</td>
        {% endfor %}
      </tr>
    </table>
  </tr>
{% endif %}
TWIG,
      ]),
    ];

    $config = $this->state->get('symfony_mailer_addons_links', []);

    $form['#tree'] = TRUE;
    $form['#attached']['library'][] = 'core/drupal.ajax';

    // Loop through each language to create a fieldset.
    foreach ($languages as $langcode => $language) {
      $form[$langcode] = [
        '#type' => 'details',
        '#title' => $this->t('Links for @language', ['@language' => $language->getName()]),
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      $links_wrapper_id = 'links-wrapper-' . $langcode;
      $form[$langcode]['links'] = [
        '#type' => 'container',
        '#prefix' => '<div id="' . $links_wrapper_id . '">',
        '#suffix' => '</div>',
        '#tree' => TRUE,
      ];

      // Get the number of link fields to show for this language.
      $num_links = $form_state->get($langcode . '_num_links') ?: count($config[$langcode] ?? []);
      if ($num_links === 0) {
        // Always show at least one empty field.
        $num_links = 1;
      }
      $form_state->set($langcode . '_num_links', $num_links);

      // Generate the link fields based on the number of links to show.
      for ($i = 0; $i < $num_links; $i++) {
        $form[$langcode]['links'][$i] = [
          '#type' => 'fieldset',
          '#attributes' => ['class' => ['container-inline', 'multilingual-link-item']],
          '#tree' => TRUE,
        ];
        $form[$langcode]['links'][$i]['title'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Title'),
          '#default_value' => $config[$langcode][$i]['title'] ?? '',
        ];

        $form[$langcode]['links'][$i]['url'] = [
          '#type' => 'textfield',
          '#title' => $this->t('URL'),
          '#default_value' => $this->getUriAsDisplayableString($config[$langcode][$i]['url'] ?? ''),
          '#element_validate' => [
            ['Drupal\\link\\Plugin\\Field\\FieldWidget\\LinkWidget', 'validateUriElement'],
          ],
          '#states' => [
            'required' => [
              ':input[name="' . $langcode . ' [links][' . $i . '][title]"]' => ['filled' => TRUE],
            ],
          ],
        ];
      }

      // Container for the action buttons.
      $form[$langcode]['actions'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['form-actions', 'clearfix']],
      ];

      // Add a button that adds more link fields via AJAX.
      $form[$langcode]['actions']['add_one'] = [
        '#type' => 'submit',
        '#value' => $this->t('Add one'),
        '#submit' => ['::addOne'],
        '#ajax' => [
          'callback' => '::addOneCallback',
          'wrapper' => $links_wrapper_id,
        ],
        // The '#name' and '#langcode' properties help the AJAX callback.
        '#name' => 'add_one_' . $langcode,
        '#langcode' => $langcode,
      ];

      // Add a button to remove the last link field,
      // visible only if there's more than one.
      if ($num_links > 1) {
        $form[$langcode]['actions']['remove_one'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove last'),
          '#submit' => ['::removeOne'],
          '#ajax' => [
            'callback' => '::addOneCallback',
            'wrapper' => $links_wrapper_id,
          ],
          '#name' => 'remove_one_' . $langcode,
          '#langcode' => $langcode,
        ];
      }
    }

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save links'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    foreach ($form_state->getValues() as $langcode => $language_values) {
      if (is_array($language_values) && isset($language_values['links'])) {
        foreach ($language_values['links'] as $i => $link_item) {
          // Check if a title is provided but the URL is missing.
          if (!empty($link_item['title']) && empty($link_item['url'])) {
            $form_state->setErrorByName($langcode . '][links][' . $i . '][url', $this->t('The URL is required if a title is provided.'));
          }
        }
      }
    }
  }

  /**
   * Submit handler for the "Add one" button.
   */
  public function addOne(array &$form, FormStateInterface $form_state) {
    // Get the language code from the triggering element's properties.
    $triggering_element = $form_state->getTriggeringElement();
    $langcode = $triggering_element['#langcode'];

    // Increment the number of links to show.
    $num_links = $form_state->get($langcode . '_num_links') + 1;
    $form_state->set($langcode . '_num_links', $num_links);

    // Rebuild the form to show the new link field.
    $form_state->setRebuild(TRUE);
  }

  /**
   * Submit handler for the "Remove last" button.
   */
  public function removeOne(array &$form, FormStateInterface $form_state) {
    // Get the language code from the triggering element's properties.
    $triggering_element = $form_state->getTriggeringElement();
    $langcode = $triggering_element['#langcode'];

    // Get the current number of links.
    $num_links = $form_state->get($langcode . '_num_links');
    // Decrement the number of links only if there is more than one.
    if ($num_links > 1) {
      $form_state->set($langcode . '_num_links', $num_links - 1);
    }

    // Rebuild the form to remove the last link field.
    $form_state->setRebuild(TRUE);
  }

  /**
   * Ajax callback for adding or removing a link field.
   */
  public function addOneCallback(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $langcode = $triggering_element['#langcode'];
    // Return the container with the updated fields.
    return $form[$langcode]['links'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $links_to_save = [];

    // Loop through the submitted values by language.
    $languages = $this->languageManager->getLanguages();

    foreach ($languages as $langcode => $language) {
      if (isset($values[$langcode]['links'])) {
        // Filter out empty link fields before saving.
        $links_to_save[$langcode] = array_filter($values[$langcode]['links'], function ($link) {
          return !empty($link['url']) || !empty($link['title']);
        });
        // Reset the array keys to be sequential.
        $links_to_save[$langcode] = array_values($links_to_save[$langcode]);
      }
    }

    // Store the cleaned data in the state system.
    $this->state->set('symfony_mailer_addons_links', $links_to_save);

    // Display a success message to the user.
    $this->messenger()->addStatus($this->t('Multilingual links have been saved.'));
  }

  /**
   * Gets the URI without the 'internal:' or 'entity:' scheme.
   *
   * @param string $uri
   *   The URI to get the displayable string for.
   *
   * @return string
   *   The displayable string for the URI.
   */
  protected function getUriAsDisplayableString($uri) {
    if (empty($uri)) {
      return $uri;
    }

    try {
      $scheme = parse_url($uri, PHP_URL_SCHEME);

      // By default, the displayable string is the URI.
      $displayable_string = $uri;

      // A different displayable string may be chosen in case of the 'internal:'
      // or 'entity:' built-in schemes.
      if ($scheme === 'internal') {
        $uri_reference = explode(':', $uri, 2)[1];
        $path = parse_url($uri, PHP_URL_PATH);
        if ($path === '/') {
          $uri_reference = '<front>' . substr($uri_reference, 1);
        }

        $displayable_string = $uri_reference;
      }
      elseif ($scheme === 'entity') {
        [$entity_type, $entity_id] = explode('/', substr($uri, 7), 2);
        if ($entity_type == 'node' && $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id)) {
          $displayable_string = EntityAutocomplete::getEntityLabels([$entity]);
        }
      }
      elseif ($scheme === 'route') {
        $displayable_string = ltrim($displayable_string, 'route:');
      }

      return $displayable_string;
    }
    catch (\InvalidArgumentException) {
      // If the URI is invalid, return it as is.
    }

    return $uri;
  }

}
