<?php

namespace Drupal\symfony_mailer_addons\Plugin\EmailAdjuster;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\symfony_mailer\EmailInterface;
use Drupal\symfony_mailer\Processor\EmailAdjusterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the Body Format Email Adjuster.
 *
 * @EmailAdjuster(
 *   id = "legacy_email_body_format",
 *   label = @Translation("Legacy Body Format Email Adjuster"),
 *   description = @Translation("Fixes the legacy email body format."),
 * )
 */
class LegacyBodyFormatEmailAdjuster extends EmailAdjusterBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructor for the LegacyEmailBodyAdjuster class.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build(EmailInterface $email) {
    $enabled = $this->configuration['enabled'] ?? FALSE;

    if (!$enabled) {
      return;
    }

    $legacy_message = $email->getParam('legacy_message');

    if (isset($legacy_message)) {
      $bodies = $email->getBody();
      foreach ($bodies as &$body) {
        if (isset($body['#type']) && $body['#type'] === 'processed_text' && !isset($body['#format'])) {
          // Check if 'email_html' text format exists before using it.
          if ($this->textFormatExists('email_html')) {
            $body['#format'] = 'email_html';
          }
          else {
            // Fall back to a default format if 'email_html' doesn't exist.
            $body['#format'] = 'plain_text';
          }
        }
      }
      $email->setBody($bodies);
    }
  }

  /**
   * Check if a text format exists.
   *
   * @param string $format_id
   *   The ID of the text format to check.
   *
   * @return bool
   *   TRUE, if the text format exists; FALSE otherwise.
   */
  protected function textFormatExists(string $format_id): bool {
    $format_storage = $this->entityTypeManager->getStorage('filter_format');
    $format = $format_storage->load($format_id);
    return $format !== NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['enabled'] = [
      '#title' => $this->t('Enable'),
      '#type' => 'checkbox',
      '#default_value' => $this->configuration['enabled'] ?? NULL,
      '#description' => $this->t('Fixes legacy mail body (if there are some HTML tags).'),
    ];

    return $form;
  }

}
