/**
 * @file
 * System Module Insights JavaScript functionality.
 */

(function ($, Drupal, drupalSettings) {
  'use strict';

  /**
   * System Module Insights behavior.
   */
  Drupal.behaviors.systemModuleInsights = {
    attach: function (context, settings) {
      var $wrapper = $('.module-viewer-wrapper', context);
      if ($wrapper.length === 0) {
        return;
      }

      var self = this;
      var ajaxUrl = drupalSettings.systemModuleInsights?.ajaxUrl || '/admin/reports/system-module-insights/ajax-filter';
      var debounceTimer = null;
      var currentFilters = this.getCurrentFilters();

      // Initialize AJAX filtering
      this.initAjaxFilters($wrapper, ajaxUrl);
      
      // Initialize column sorting
      this.initColumnSorting($wrapper, ajaxUrl);
      
      // Initialize export functionality
      this.initExportFunctionality($wrapper);
      
      // Initialize tooltips
      this.initTooltips($wrapper);
    },

    /**
     * Get current filter values from the form.
     */
    getCurrentFilters: function () {
      var filters = {};
      var $form = $('.module-viewer-filters');
      
      if ($form.length) {
        filters.search = $form.find('input[name="search"]').val() || '';
        filters.status = $form.find('select[name="status"]').val() || '';
        filters.package = $form.find('select[name="package"]').val() || '';
        filters.custom_only = $form.find('input[name="custom_only"]:checked').length ? '1' : '';
        filters.unused_only = $form.find('input[name="unused_only"]:checked').length ? '1' : '';
      }
      
      return filters;
    },

    /**
     * Initialize AJAX filtering functionality.
     */
    initAjaxFilters: function ($wrapper, ajaxUrl) {
      var self = this;
      
      // Handle search input with debouncing
      $('.ajax-filter[data-filter-type="search"]').on('input', function () {
        clearTimeout(self.debounceTimer);
        self.debounceTimer = setTimeout(function () {
          self.performAjaxFilter(ajaxUrl);
        }, 500); // 500ms debounce
      });

      // Handle other filter changes immediately
      $('.ajax-filter').not('[data-filter-type="search"]').on('change', function () {
        clearTimeout(self.debounceTimer);
        self.performAjaxFilter(ajaxUrl);
      });
    },

    /**
     * Initialize column sorting functionality.
     */
    initColumnSorting: function ($wrapper, ajaxUrl) {
      var self = this;
      
      $wrapper.on('click', '.sortable-header a.sort-link', function (e) {
        e.preventDefault();
        
        var $header = $(this).closest('th');
        var sortField = $header.data('sort-field');
        var sortOrder = $header.data('sort-order');
        
        // Update current sort in settings
        drupalSettings.systemModuleInsights.currentSort = sortField;
        drupalSettings.systemModuleInsights.currentOrder = sortOrder;
        
        // Perform AJAX request with sort parameters
        self.performAjaxFilter(ajaxUrl, {
          sort: sortField,
          order: sortOrder
        });
      });
    },

    /**
     * Initialize export functionality.
     */
    initExportFunctionality: function ($wrapper) {
      var self = this;
      
      var columnSelector = '#edit-columns input[type="checkbox"], .export-columns-checkboxes input[type="checkbox"]';

      // Handle export link clicks with selected columns
      $(document).on('click', '.export-link', function (e) {
        e.preventDefault();
        var $link = $(this);
        var baseUrl = $link.attr('href');
        var selectedColumns = [];
        
        // Get selected columns from the form
        $(columnSelector).filter(':checked').each(function () {
          selectedColumns.push($(this).val());
        });
        
        // If no columns are selected, export all columns
        var finalUrl = baseUrl;
        if (selectedColumns.length > 0) {
          var separator = baseUrl.indexOf('?') !== -1 ? '&' : '?';
          var columnsParam = selectedColumns.map(function (col) {
            return 'columns[]=' + encodeURIComponent(col);
          }).join('&');
          
          finalUrl = baseUrl + separator + columnsParam;
        }
        
        // Open the export URL
        window.open(finalUrl, '_blank');
      });

      // Export column selection helper
      $(document).on('change', columnSelector, function () {
        self.updateExportColumnInfo();
      });
      
      // Initialize column info display
      this.updateExportColumnInfo();
    },
    
    /**
     * Update export column information display.
     */
    updateExportColumnInfo: function() {
      var columnSelector = '#edit-columns input[type="checkbox"], .export-columns-checkboxes input[type="checkbox"]';
      var $checkboxes = $(columnSelector);
      var checkedCount = $checkboxes.filter(':checked').length;
      var totalCount = $checkboxes.length;
      
      var infoText = '';
      if (checkedCount === 0) {
        infoText = 'All columns will be exported';
      } else {
        infoText = checkedCount + ' of ' + totalCount + ' columns selected';
      }
      
      var $infoElements = $('.export-column-info');
      if ($infoElements.length === 0) {
        var $infoDiv = $('<div class="export-column-info form-item"></div>').text(infoText);
        $('#edit-export-options-columns, .export-options .form-checkboxes').after($infoDiv);
      } else {
        $infoElements.text(infoText);
      }
    },

    /**
     * Initialize tooltips for various elements.
     */
    initTooltips: function ($wrapper) {
      // Check if jQuery UI tooltip is available
      if (typeof $.fn.tooltip === 'undefined') {
        // Fallback: use native browser tooltips (title attribute)
        // No additional initialization needed
        return;
      }
      
      // Initialize tooltips for elements with title attributes
      $wrapper.find('[title]').each(function () {
        var $element = $(this);
        var title = $element.attr('title');
        
        if (title && title.length > 0) {
          try {
            $element.tooltip({
              content: title,
              position: {
                my: 'center bottom-10',
                at: 'center top',
                using: function (position, feedback) {
                  $(this).css(position);
                  $('<div>')
                    .addClass('tooltip-arrow')
                    .addClass(feedback.vertical)
                    .addClass(feedback.horizontal)
                    .appendTo(this);
                }
              }
            });
          } catch (e) {
            // If tooltip initialization fails, just keep the native title attribute
            console.warn('Tooltip initialization failed:', e);
          }
        }
      });
    },

    /**
     * Perform AJAX filter request.
     */
    performAjaxFilter: function (ajaxUrl, additionalParams) {
      var self = this;
      var filters = this.getCurrentFilters();
      var params = $.extend({}, filters, additionalParams || {});
      
      // Add current sort parameters if they exist
      if (drupalSettings.systemModuleInsights.currentSort) {
        params.sort = drupalSettings.systemModuleInsights.currentSort;
        params.order = drupalSettings.systemModuleInsights.currentOrder;
      }
      
      // Show loading overlay
      this.showLoadingOverlay();
      
      // Perform AJAX request
      $.ajax({
        url: ajaxUrl,
        type: 'GET',
        data: params,
        dataType: 'json',
        success: function (response) {
          // Process AJAX response
          if (response && response.length > 0) {
            response.forEach(function (command) {
              if (command.command === 'replace') {
                $(command.selector).replaceWith(command.data);
              }
            });
            
            // Re-attach behaviors to new content
            Drupal.attachBehaviors($('.module-viewer-wrapper'));
          }
          
          self.hideLoadingOverlay();
        },
        error: function (xhr, status, error) {
          console.error('AJAX filter request failed:', error);
          self.hideLoadingOverlay();
          self.showErrorMessage('Failed to filter modules. Please try again.');
        }
      });
      
      // Update URL with current filters (for bookmarking/sharing)
      this.updateUrl(params);
    },

    /**
     * Show loading overlay.
     */
    showLoadingOverlay: function () {
      $('.ajax-loading-overlay').show();
    },

    /**
     * Hide loading overlay.
     */
    hideLoadingOverlay: function () {
      $('.ajax-loading-overlay').hide();
    },

    /**
     * Show error message.
     */
    showErrorMessage: function (message) {
      var $errorDiv = $('<div class="messages messages--error" role="contentinfo">')
        .text(message)
        .hide();
      
      $('.module-viewer-wrapper').prepend($errorDiv);
      $errorDiv.slideDown();
      
      // Remove error message after 5 seconds
      setTimeout(function () {
        $errorDiv.slideUp(function () {
          $errorDiv.remove();
        });
      }, 5000);
    },

    /**
     * Update URL with current filter parameters.
     */
    updateUrl: function (params) {
      if (window.history && window.history.pushState) {
        var url = new URL(window.location);
        
        // Clear existing query parameters related to filtering
        var filterParams = ['search', 'status', 'package', 'custom_only', 'unused_only', 'sort', 'order'];
        filterParams.forEach(function (param) {
          url.searchParams.delete(param);
        });
        
        // Add new parameters
        Object.keys(params).forEach(function (key) {
          if (params[key] && params[key] !== '') {
            url.searchParams.set(key, params[key]);
          }
        });
        
        window.history.pushState({}, '', url);
      }
    }
  };

  /**
   * Enhanced table functionality.
   */
  Drupal.behaviors.systemModuleInsightsTable = {
    attach: function (context, settings) {
      var $table = $('.module-viewer-table', context);
      if ($table.length === 0) {
        return;
      }

      // Initialize row hover effects
      this.initRowHoverEffects($table);
      
      // Initialize expandable content
      this.initExpandableContent($table);
      
      // Initialize keyboard navigation
      this.initKeyboardNavigation($table);
    },

    /**
     * Initialize row hover effects and interactions.
     */
    initRowHoverEffects: function ($table) {
      $table.on('mouseenter', '.module-row', function () {
        $(this).addClass('row-hover');
      }).on('mouseleave', '.module-row', function () {
        $(this).removeClass('row-hover');
      });
    },

    /**
     * Initialize expandable content for truncated descriptions and lists.
     */
    initExpandableContent: function ($table) {
      // Handle expandable descriptions
      $table.on('click', '.description-text[title]', function () {
        var $desc = $(this);
        var fullText = $desc.attr('title');
        var currentText = $desc.text();
        
        if (currentText.endsWith('...')) {
          $desc.text(fullText).removeClass('truncated');
        } else {
          var truncated = fullText.length > 80 ? fullText.substring(0, 80) + '...' : fullText;
          $desc.text(truncated).addClass('truncated');
        }
      });
      
      // Handle expandable dependency/dependent lists
      $table.on('click', '.more-indicator', function (e) {
        e.preventDefault();
        var $indicator = $(this);
        var $parent = $indicator.closest('td');
        var $list = $indicator.parent('.dependency-list, .dependent-list, .metric-details');
        var fullList = $indicator.attr('title');
        
        if ($list.hasClass('expanded')) {
          // Collapse - show only first 3 items
          var items = fullList.split(', ');
          var truncatedItems = items.slice(0, 3);
          
          // Determine the correct item class based on parent
          var itemClass = 'dependency-item';
          if ($parent.hasClass('dependents')) {
            itemClass = 'dependent-item';
          } else if ($parent.hasClass('fields')) {
            itemClass = 'field-item';
          }
          
          $list.html(truncatedItems.map(function (item) {
            return '<span class="' + itemClass + '">' + item.trim() + '</span>';
          }).join(' '));
          
          // Re-add the more indicator
          var moreCount = items.length - 3;
          var $newIndicator = $('<span class="more-indicator" title="' + fullList + '">+' + moreCount + ' more</span>');
          $list.append(' ').append($newIndicator);
          
          $list.removeClass('expanded');
        } else {
          // Expand - show all items
          var items = fullList.split(', ');
          
          // Determine the correct item class based on parent
          var itemClass = 'dependency-item';
          if ($parent.hasClass('dependents')) {
            itemClass = 'dependent-item';
          } else if ($parent.hasClass('fields')) {
            itemClass = 'field-item';
          }
          
          $list.html(items.map(function (item) {
            return '<span class="' + itemClass + '">' + item.trim() + '</span>';
          }).join(' '));
          
          // Add the show less indicator
          var $lessIndicator = $('<span class="more-indicator show-less" title="' + fullList + '">Show less</span>');
          $list.append(' ').append($lessIndicator);
          
          $list.addClass('expanded');
        }
      });
    },

    /**
     * Initialize keyboard navigation for accessibility.
     */
    initKeyboardNavigation: function ($table) {
      // Make table rows focusable
      $table.find('.module-row').attr('tabindex', '0');
      
      // Handle keyboard navigation
      $table.on('keydown', '.module-row', function (e) {
        var $current = $(this);
        var $next, $prev;
        
        switch (e.which) {
          case 38: // Up arrow
            $prev = $current.prev('.module-row');
            if ($prev.length) {
              $prev.focus();
              e.preventDefault();
            }
            break;
            
          case 40: // Down arrow
            $next = $current.next('.module-row');
            if ($next.length) {
              $next.focus();
              e.preventDefault();
            }
            break;
            
          case 13: // Enter key
          case 32: // Space key
            // Toggle row selection or trigger action
            $current.toggleClass('selected');
            e.preventDefault();
            break;
        }
      });
    }
  };

})(jQuery, Drupal, drupalSettings);
