<?php

namespace Drupal\system_module_insights\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\system_module_insights\Service\ModuleAnalyzer;
use Drupal\system_module_insights\Service\ModuleExporter;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;

/**
 * Controller for System Module Insights functionality.
 */
class ModuleViewerController extends ControllerBase {

  /**
   * The module analyzer service.
   *
   * @var \Drupal\system_module_insights\Service\ModuleAnalyzer
   */
  protected $moduleAnalyzer;

  /**
   * The module exporter service.
   *
   * @var \Drupal\system_module_insights\Service\ModuleExporter
   */
  protected $moduleExporter;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a ModuleViewerController object.
   */
  public function __construct(
    ModuleAnalyzer $module_analyzer,
    ModuleExporter $module_exporter,
    RendererInterface $renderer
  ) {
    $this->moduleAnalyzer = $module_analyzer;
    $this->moduleExporter = $module_exporter;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('system_module_insights.analyzer'),
      $container->get('system_module_insights.exporter'),
      $container->get('renderer')
    );
  }

  /**
   * Display the main system module insights page.
   */
  public function view(Request $request) {
    $build = [];

    // Get filters from request
    $filters = [
      'search' => $request->query->get('search', ''),
      'status' => $request->query->get('status', ''),
      'package' => $request->query->get('package', ''),
      'custom_only' => $request->query->get('custom_only', FALSE),
      'unused_only' => $request->query->get('unused_only', FALSE),
    ];

    // Get sorting parameters
    $sort_field = $request->query->get('sort', 'name');
    $sort_direction = $request->query->get('order', 'asc');

    // Build the filter form
    $form = $this->formBuilder()->getForm(
      'Drupal\system_module_insights\Form\ModuleViewerFilterForm',
      $filters
    );
    $build['filters'] = $form;

    // Get module data
    $modules_data = $this->moduleAnalyzer->getModuleData($filters);

    // Sort the data
    $modules_data = $this->sortModuleData($modules_data, $sort_field, $sort_direction);

    // Create export links
    $export_links = $this->buildExportLinks($filters);

    // Build the main table
    $build['module_table'] = [
      '#theme' => 'system_module_insights_table',
      '#modules' => $modules_data,
      '#header' => $this->getTableHeader($sort_field, $sort_direction),
      '#filters' => $filters,
      '#export_links' => $export_links,
      '#attached' => [
        'library' => ['system_module_insights/system_module_insights'],
        'drupalSettings' => [
          'systemModuleInsights' => [
            'ajaxUrl' => '/admin/reports/system-module-insights/ajax-filter',
            'currentSort' => $sort_field,
            'currentOrder' => $sort_direction,
          ],
        ],
      ],
    ];

    // Add summary information
    $build['summary'] = [
      '#markup' => $this->buildSummary($modules_data, $filters),
      '#weight' => -10,
    ];

    return $build;
  }

  /**
   * AJAX callback for filtering modules.
   */
  public function ajaxFilter(Request $request) {
    $response = new AjaxResponse();

    // Get filters from request
    $filters = [
      'search' => $request->query->get('search', ''),
      'status' => $request->query->get('status', ''),
      'package' => $request->query->get('package', ''),
      'custom_only' => $request->query->get('custom_only', FALSE),
      'unused_only' => $request->query->get('unused_only', FALSE),
    ];

    // Get sorting parameters
    $sort_field = $request->query->get('sort', 'name');
    $sort_direction = $request->query->get('order', 'asc');

    // Get module data
    $modules_data = $this->moduleAnalyzer->getModuleData($filters);
    $modules_data = $this->sortModuleData($modules_data, $sort_field, $sort_direction);

    // Create export links
    $export_links = $this->buildExportLinks($filters);

    // Render the updated table
    $table_build = [
      '#theme' => 'system_module_insights_table',
      '#modules' => $modules_data,
      '#header' => $this->getTableHeader($sort_field, $sort_direction),
      '#filters' => $filters,
      '#export_links' => $export_links,
    ];

    $rendered_table = $this->renderer->render($table_build);
    $response->addCommand(new ReplaceCommand('#module-viewer-table', $rendered_table));

    // Update summary
    $summary_build = [
      '#markup' => $this->buildSummary($modules_data, $filters),
    ];
    $rendered_summary = $this->renderer->render($summary_build);
    $response->addCommand(new ReplaceCommand('.module-viewer-summary', $rendered_summary));

    return $response;
  }

  /**
   * Export module data.
   */
  public function export(Request $request, $format) {
    try {
      // Get filters from request - handle each parameter safely
      $filters = [];
      $filters['search'] = $request->query->has('search') ? $request->query->get('search') : '';
      $filters['status'] = $request->query->has('status') ? $request->query->get('status') : '';
      $filters['package'] = $request->query->has('package') ? $request->query->get('package') : '';
      $filters['custom_only'] = $request->query->has('custom_only') ? (bool)$request->query->get('custom_only') : FALSE;
      $filters['unused_only'] = $request->query->has('unused_only') ? (bool)$request->query->get('unused_only') : FALSE;

      // Get selected columns from request - handle array parameter safely
      $selected_columns = [];
      if ($request->query->has('columns')) {
        $columns_param = $request->query->get('columns');
        if (is_array($columns_param)) {
          // Filter out empty values and ensure valid column names
          $selected_columns = array_filter($columns_param, function($column) {
            return !empty($column) && is_string($column);
          });
        }
      }
      
      // Log for debugging
      \Drupal::logger('system_module_insights')->debug('Export columns parameter: @columns', [
        '@columns' => print_r($selected_columns, TRUE)
      ]);

      // Get module data
      $modules_data = $this->moduleAnalyzer->getModuleData($filters);

      // Export based on format
      switch ($format) {
        case 'csv':
          return $this->moduleExporter->exportToCsv($modules_data, $selected_columns);

        case 'json':
          return $this->moduleExporter->exportToJson($modules_data, $selected_columns);

        case 'pdf':
          return $this->moduleExporter->exportToPdf($modules_data, $selected_columns);

        default:
          throw new \InvalidArgumentException('Unsupported export format: ' . $format);
      }
    }
    catch (\Exception $e) {
      // Log the error for debugging
      \Drupal::logger('system_module_insights')->error('Export error: @message', ['@message' => $e->getMessage()]);
      
      // Return a simple error response
      return new Response(
        'Export failed: ' . $e->getMessage(), 
        500,
        ['Content-Type' => 'text/plain']
      );
    }
  }

  /**
   * Sort module data.
   */
  protected function sortModuleData(array $modules_data, $sort_field, $sort_direction) {
    uasort($modules_data, function ($a, $b) use ($sort_field, $sort_direction) {
      $value_a = $this->getSortValue($a, $sort_field);
      $value_b = $this->getSortValue($b, $sort_field);

      if ($value_a == $value_b) {
        return 0;
      }

      $result = ($value_a < $value_b) ? -1 : 1;
      return ($sort_direction === 'desc') ? -$result : $result;
    });

    return $modules_data;
  }

  /**
   * Get sort value for a field.
   */
  protected function getSortValue($module_data, $sort_field) {
    switch ($sort_field) {
      case 'fields':
        return $module_data['usage_metrics']['fields'];

      case 'views':
        return $module_data['usage_metrics']['views'];

      case 'dependencies_count':
        return count($module_data['dependencies']);

      case 'dependents_count':
        return count($module_data['dependents']);

      default:
        return strtolower($module_data[$sort_field] ?? '');
    }
  }

  /**
   * Get table header with sorting links.
   */
  protected function getTableHeader($current_sort, $current_order) {
    $headers = [
      'name' => $this->t('Module Name'),
      'display_name' => $this->t('Display Name'),
      'description' => $this->t('Description'),
      'package' => $this->t('Package'),
      'version' => $this->t('Version'),
      'status' => $this->t('Status'),
      'fields' => $this->t('Fields'),
      'views' => $this->t('Views'),
      'dependencies' => $this->t('Dependencies'),
      'dependents' => $this->t('Dependents'),
    ];

    // Add sort indicators and links
    foreach ($headers as $field => &$header) {
      $new_order = ($current_sort == $field && $current_order == 'asc') ? 'desc' : 'asc';
      $indicator = '';
      
      if ($current_sort == $field) {
        $indicator = ($current_order == 'asc') ? ' ↑' : ' ↓';
      }

      $header = [
        'data' => $header . $indicator,
        'class' => ['sortable-header'],
        'data-sort-field' => $field,
        'data-sort-order' => $new_order,
      ];
    }

    return $headers;
  }

  /**
   * Build export links.
   */
  protected function buildExportLinks($filters) {
    // Clean filters to only include non-empty values
    $clean_filters = [];
    foreach ($filters as $key => $value) {
      if (!empty($value) && $value !== 'all') {
        $clean_filters[$key] = $value;
      }
    }
    
    return [
      'csv' => [
        'title' => $this->t('Export CSV'),
        'url' => Url::fromRoute('system_module_insights.export', ['format' => 'csv'], ['query' => $clean_filters])->toString(),
        'attributes' => ['class' => ['button', 'export-link']],
      ],
      'json' => [
        'title' => $this->t('Export JSON'),
        'url' => Url::fromRoute('system_module_insights.export', ['format' => 'json'], ['query' => $clean_filters])->toString(),
        'attributes' => ['class' => ['button', 'export-link']],
      ],
      'pdf' => [
        'title' => $this->t('Export HTML/PDF'),
        'url' => Url::fromRoute('system_module_insights.export', ['format' => 'pdf'], ['query' => $clean_filters])->toString(),
        'attributes' => ['class' => ['button', 'export-link']],
      ],
    ];
  }

  /**
   * Build summary information.
   */
  protected function buildSummary($modules_data, $filters = []) {
    $total = count($modules_data);
    $enabled = 0;
    $custom = 0;
    $unused = 0;
    
    foreach ($modules_data as $module) {
      if ($module['status'] === 'enabled') {
        $enabled++;
      }
      if ($module['is_custom']) {
        $custom++;
      }
      // Remove usage score check for unused modules since we no longer have usage scores
      // if ($module['usage_score'] == 0) {
      //   $unused++;
      // }
    }

    $summary = '<div class="module-viewer-summary">';
    $summary .= '<h3>' . $this->t('Summary') . '</h3>';
    
    // Show filter information if filters are active
    $active_filters = array_filter($filters);
    if (!empty($active_filters)) {
      $summary .= '<div class="filter-info">';
      $summary .= '<strong>' . $this->t('Active Filters:') . '</strong> ';
      $filter_descriptions = [];
      
      if (!empty($filters['search'])) {
        $filter_descriptions[] = $this->t('Search: "@search"', ['@search' => $filters['search']]);
      }
      if (!empty($filters['status']) && $filters['status'] !== 'all') {
        $filter_descriptions[] = $this->t('Status: @status', ['@status' => ucfirst($filters['status'])]);
      }
      if (!empty($filters['package']) && $filters['package'] !== 'all') {
        $filter_descriptions[] = $this->t('Package: @package', ['@package' => $filters['package']]);
      }
      if (!empty($filters['custom_only'])) {
        $filter_descriptions[] = $this->t('Custom modules only');
      }
      if (!empty($filters['unused_only'])) {
        $filter_descriptions[] = $this->t('Unused modules only');
      }
      
      $summary .= implode(', ', $filter_descriptions);
      $summary .= '</div>';
    }
    
    $summary .= '<div class="summary-stats">';
    $summary .= '<span class="stat"><strong>' . $total . '</strong> ' . 
                ($active_filters ? $this->t('Matching Modules') : $this->t('Total Modules')) . '</span> | ';
    $summary .= '<span class="stat"><strong>' . $enabled . '</strong> ' . $this->t('Enabled') . '</span> | ';
    $summary .= '<span class="stat"><strong>' . $custom . '</strong> ' . $this->t('Custom') . '</span>';
    $summary .= '</div></div>';

    return $summary;
  }

}
