<?php

namespace Drupal\system_module_insights\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\system_module_insights\Service\ModuleAnalyzer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filter form for System Module Insights.
 */
class ModuleViewerFilterForm extends FormBase {

  /**
   * The module analyzer service.
   *
   * @var \Drupal\system_module_insights\Service\ModuleAnalyzer
   */
  protected $moduleAnalyzer;

  /**
   * Constructs a ModuleViewerFilterForm object.
   */
  public function __construct(ModuleAnalyzer $module_analyzer) {
    $this->moduleAnalyzer = $module_analyzer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('system_module_insights.analyzer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'system_module_insights_filter_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $filters = []) {
    $form['#attributes']['class'][] = 'module-viewer-filters';
    $form['#attributes']['data-drupal-selector'] = 'module-viewer-filter-form';

    // Search field
    $form['search'] = [
      '#type' => 'search',
      '#title' => $this->t('Search modules'),
      '#placeholder' => $this->t('Module name or description...'),
      '#default_value' => $filters['search'] ?? '',
      '#attributes' => [
        'class' => ['ajax-filter'],
        'data-filter-type' => 'search',
      ],
    ];

    // Status filter
    $form['status'] = [
      '#type' => 'select',
      '#title' => $this->t('Status'),
      '#options' => [
        '' => $this->t('- Any Status -'),
        'enabled' => $this->t('Enabled'),
        'installed' => $this->t('Installed (Disabled)'),
        'available' => $this->t('Available (Not Installed)'),
      ],
      '#default_value' => $filters['status'] ?? '',
      '#attributes' => [
        'class' => ['ajax-filter'],
        'data-filter-type' => 'status',
      ],
    ];

    // Package filter
    $packages = $this->moduleAnalyzer->getAvailablePackages();
    $package_options = ['' => $this->t('- Any Package -')] + $packages;
    
    $form['package'] = [
      '#type' => 'select',
      '#title' => $this->t('Package'),
      '#options' => $package_options,
      '#default_value' => $filters['package'] ?? '',
      '#attributes' => [
        'class' => ['ajax-filter'],
        'data-filter-type' => 'package',
      ],
    ];

    // Quick filters
    $form['quick_filters'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Quick Filters'),
      '#collapsible' => FALSE,
      '#attributes' => ['class' => ['quick-filters']],
    ];

    $form['quick_filters']['custom_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show only custom modules'),
      '#default_value' => $filters['custom_only'] ?? FALSE,
      '#attributes' => [
        'class' => ['ajax-filter'],
        'data-filter-type' => 'custom_only',
      ],
    ];

    $form['quick_filters']['unused_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show only unused modules'),
      '#default_value' => $filters['unused_only'] ?? FALSE,
      '#attributes' => [
        'class' => ['ajax-filter'],
        'data-filter-type' => 'unused_only',
      ],
    ];

    // Export options
    $form['export_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Export Options'),
      '#open' => FALSE,
      '#attributes' => ['class' => ['export-options']],
    ];

    $form['export_options']['columns'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select columns to export'),
      '#options' => [
        'name' => $this->t('Module Name'),
        'display_name' => $this->t('Display Name'),
        'description' => $this->t('Description'),
        'package' => $this->t('Package'),
        'version' => $this->t('Version'),
        'status' => $this->t('Status'),
        'dependencies' => $this->t('Dependencies'),
        'dependents' => $this->t('Dependents'),
        'fields' => $this->t('Fields'),
        'views' => $this->t('Views'),
        'config_items' => $this->t('Config Items'),
        'permissions' => $this->t('Permissions'),
        'is_core' => $this->t('Is Core'),
        'is_custom' => $this->t('Is Custom'),
      ],
      '#default_value' => [],
      '#description' => $this->t('Select which columns to include in exports. Leave empty to export all columns.'),
      '#attributes' => ['class' => ['export-columns-checkboxes']],
    ];
    
    $form['export_options']['column_info'] = [
      '#type' => 'markup',
      '#markup' => '<div class="export-column-info">All columns will be exported</div>',
    ];

    // Form actions
    $form['actions'] = [
      '#type' => 'actions',
      '#attributes' => ['class' => ['filter-actions']],
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Apply Filters'),
      '#attributes' => ['class' => ['button--primary']],
    ];

    $form['actions']['reset'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset Filters'),
      '#submit' => ['::resetForm'],
      '#attributes' => ['class' => ['button--secondary']],
      '#limit_validation_errors' => [],
    ];

    // Add help text
    $form['help'] = [
      '#type' => 'details',
      '#title' => $this->t('Help & Tips'),
      '#open' => FALSE,
      '#weight' => 100,
    ];

    $form['help']['tips'] = [
      '#markup' => '<div class="module-viewer-help">' .
        '<h4>' . $this->t('Usage Tips:') . '</h4>' .
        '<ul>' .
        '<li>' . $this->t('<strong>Usage Score:</strong> ★★★★★ indicates how extensively a module is used (content types, fields, views, etc.)') . '</li>' .
        '<li>' . $this->t('<strong>Custom Modules:</strong> Highlighted in blue, located in modules/custom/') . '</li>' .
        '<li>' . $this->t('<strong>Unused Modules:</strong> Enabled modules with zero usage score, highlighted in yellow') . '</li>' .
        '<li>' . $this->t('<strong>Sorting:</strong> Click column headers to sort. Click again to reverse order.') . '</li>' .
        '<li>' . $this->t('<strong>Real-time Search:</strong> Type in the search box for instant filtering') . '</li>' .
        '<li>' . $this->t('<strong>Export:</strong> Select specific columns and format for custom reports') . '</li>' .
        '</ul>' .
        '</div>',
    ];

    // Attach library for AJAX functionality
    $form['#attached']['library'][] = 'system_module_insights/system_module_insights';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get form values
    $values = $form_state->getValues();
    
    // Build query parameters
    $query = [];
    
    if (!empty($values['search'])) {
      $query['search'] = $values['search'];
    }
    
    if (!empty($values['status'])) {
      $query['status'] = $values['status'];
    }
    
    if (!empty($values['package'])) {
      $query['package'] = $values['package'];
    }
    
    if (!empty($values['custom_only'])) {
      $query['custom_only'] = 1;
    }
    
    if (!empty($values['unused_only'])) {
      $query['unused_only'] = 1;
    }

    // Redirect with query parameters
    $form_state->setRedirect('system_module_insights.view', [], ['query' => $query]);
  }

  /**
   * Reset form submit handler.
   */
  public function resetForm(array &$form, FormStateInterface $form_state) {
    $form_state->setRedirect('system_module_insights.view');
  }

}