<?php

namespace Drupal\system_module_insights\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for System Module Insights settings.
 */
class ModuleViewerSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['system_module_insights.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'system_module_insights_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('system_module_insights.settings');

    $form['display'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display Settings'),
    ];

    $form['display']['items_per_page'] = [
      '#type' => 'number',
      '#title' => $this->t('Items per page'),
      '#default_value' => $config->get('items_per_page') ?: 50,
      '#min' => 10,
      '#max' => 500,
      '#description' => $this->t('Number of modules to display per page (10-500).'),
    ];

    $form['display']['default_columns'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Default visible columns'),
      '#options' => [
        'name' => $this->t('Module Name'),
        'display_name' => $this->t('Display Name'),
        'description' => $this->t('Description'),
        'package' => $this->t('Package'),
        'version' => $this->t('Version'),
        'status' => $this->t('Status'),
        'usage_score' => $this->t('Usage Score'),
        'content_types' => $this->t('Content Types'),
        'fields' => $this->t('Fields'),
        'views' => $this->t('Views'),
        'dependencies' => $this->t('Dependencies'),
        'dependents' => $this->t('Dependents'),
        'path' => $this->t('Path'),
      ],
      '#default_value' => $config->get('default_columns') ?: [
        'name',
        'status',
        'package',
        'usage_score',
        'dependencies',
        'dependents',
      ],
      '#description' => $this->t('Select which columns should be visible by default.'),
    ];

    $form['performance'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Performance Settings'),
    ];

    $form['performance']['cache_duration'] = [
      '#type' => 'select',
      '#title' => $this->t('Cache duration'),
      '#options' => [
        0 => $this->t('No caching'),
        300 => $this->t('5 minutes'),
        900 => $this->t('15 minutes'),
        1800 => $this->t('30 minutes'),
        3600 => $this->t('1 hour'),
        7200 => $this->t('2 hours'),
        21600 => $this->t('6 hours'),
        86400 => $this->t('1 day'),
      ],
      '#default_value' => $config->get('cache_duration') ?: 900,
      '#description' => $this->t('How long to cache module analysis data. Longer cache improves performance but may show outdated information.'),
    ];

    $form['performance']['enable_ajax'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable AJAX filtering'),
      '#default_value' => $config->get('enable_ajax') ?? TRUE,
      '#description' => $this->t('Enable real-time filtering without page reloads. Disable if experiencing JavaScript issues.'),
    ];

    $form['analysis'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Analysis Settings'),
    ];

    $form['analysis']['include_test_modules'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include test modules'),
      '#default_value' => $config->get('include_test_modules') ?? FALSE,
      '#description' => $this->t('Include modules in test directories in the analysis.'),
    ];

    $form['analysis']['deep_dependency_scan'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Deep dependency scanning'),
      '#default_value' => $config->get('deep_dependency_scan') ?? TRUE,
      '#description' => $this->t('Perform detailed analysis of module dependencies and configuration. May slow down the analysis.'),
    ];

    $form['analysis']['usage_calculation_method'] = [
      '#type' => 'select',
      '#title' => $this->t('Usage score calculation method'),
      '#options' => [
        'balanced' => $this->t('Balanced (default)'),
        'content_focused' => $this->t('Content focused'),
        'config_focused' => $this->t('Configuration focused'),
        'comprehensive' => $this->t('Comprehensive'),
      ],
      '#default_value' => $config->get('usage_calculation_method') ?: 'balanced',
      '#description' => $this->t('How to calculate the usage score for modules.'),
    ];

    $form['export'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Export Settings'),
    ];

    $form['export']['max_export_items'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum export items'),
      '#default_value' => $config->get('max_export_items') ?: 1000,
      '#min' => 100,
      '#max' => 10000,
      '#description' => $this->t('Maximum number of items to include in exports (100-10000).'),
    ];

    $form['export']['include_metadata'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include metadata in exports'),
      '#default_value' => $config->get('include_metadata') ?? TRUE,
      '#description' => $this->t('Include export timestamp and summary information in export files.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    $this->config('system_module_insights.settings')
      ->set('items_per_page', $values['items_per_page'])
      ->set('default_columns', array_filter($values['default_columns']))
      ->set('cache_duration', $values['cache_duration'])
      ->set('enable_ajax', $values['enable_ajax'])
      ->set('include_test_modules', $values['include_test_modules'])
      ->set('deep_dependency_scan', $values['deep_dependency_scan'])
      ->set('usage_calculation_method', $values['usage_calculation_method'])
      ->set('max_export_items', $values['max_export_items'])
      ->set('include_metadata', $values['include_metadata'])
      ->save();

    parent::submitForm($form, $form_state);
  }

}