<?php

namespace Drupal\system_module_insights\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ExtensionList;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for analyzing module information and dependencies.
 */
class ModuleAnalyzer {

  use StringTranslationTrait;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ExtensionList
   */
  protected $moduleList;

  /**
   * The configuration storage.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $configStorage;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a ModuleAnalyzer object.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
    ExtensionList $module_list,
    StorageInterface $config_storage,
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    Connection $database
  ) {
    $this->moduleHandler = $module_handler;
    $this->moduleList = $module_list;
    $this->configStorage = $config_storage;
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->database = $database;
  }

  /**
   * Get comprehensive module analysis data.
   *
   * @param array $filters
   *   Filter criteria.
   *
   * @return array
   *   Array of module data.
   */
  public function getModuleData(array $filters = []) {
    $modules_data = [];
    $all_modules = $this->moduleList->getAllAvailableInfo();
    $installed_modules = $this->moduleList->getAllInstalledInfo();

    foreach ($all_modules as $module_name => $module_info) {
      $is_installed = isset($installed_modules[$module_name]);
      $is_enabled = $this->moduleHandler->moduleExists($module_name);

      // Apply basic filters early to improve performance
      if (!$this->passesBasicFilters($module_name, $module_info, $is_enabled, $filters)) {
        continue;
      }

      $module_data = [
        'name' => $module_name,
        'display_name' => $module_info['name'] ?? $module_name,
        'description' => $module_info['description'] ?? '',
        'package' => $module_info['package'] ?? 'Other',
        'version' => $module_info['version'] ?? 'Unknown',
        'status' => $is_enabled ? 'enabled' : ($is_installed ? 'installed' : 'available'),
        'is_core' => strpos($module_name, 'core/') !== FALSE || in_array($module_info['package'] ?? '', ['Core', 'Core (Experimental)']),
        'is_custom' => $this->isCustomModule($module_name),
        'path' => $this->getModulePath($module_name),
        'dependencies' => $this->getModuleDependencies($module_name, $module_info),
        'dependents' => $this->getModuleDependents($module_name),
        'usage_metrics' => $this->getUsageMetrics($module_name),
        'configuration' => $this->getConfigurationData($module_name),
        'permissions' => $this->getModulePermissions($module_name),
        'documentation_links' => $this->getDocumentationLinks($module_name, $module_info),
      ];

      // Note: usage_score removed - was previously calculated here
      
      // Add status classes for highlighting
      $module_data['status_class'] = $this->getStatusClass($module_data);

      // Apply advanced filters that require calculated data
      if (!$this->passesAdvancedFilters($module_data, $filters)) {
        continue;
      }

      $modules_data[$module_name] = $module_data;
    }

    return $modules_data;
  }

  /**
   * Check if module passes basic filters (before calculation-heavy operations).
   */
  protected function passesBasicFilters($module_name, $module_info, $is_enabled, $filters) {
    $all_installed = $this->moduleList->getAllInstalledInfo();
    $is_installed = isset($all_installed[$module_name]);

    // Search filter - search in name, display name, and description
    if (!empty($filters['search'])) {
      $search = strtolower($filters['search']);
      $searchable_text = strtolower($module_name . ' ' . 
                                   ($module_info['name'] ?? '') . ' ' . 
                                   ($module_info['description'] ?? ''));
      if (strpos($searchable_text, $search) === FALSE) {
        return FALSE;
      }
    }

    // Status filter
    if (!empty($filters['status']) && $filters['status'] !== 'all') {
      $current_status = $is_enabled ? 'enabled' : ($is_installed ? 'installed' : 'available');
      if ($filters['status'] !== $current_status) {
        return FALSE;
      }
    }

    // Package filter
    if (!empty($filters['package']) && $filters['package'] !== 'all') {
      $package = $module_info['package'] ?? 'Other';
      if ($package !== $filters['package']) {
        return FALSE;
      }
    }

    // Custom modules filter
    if (!empty($filters['custom_only'])) {
      if (!$this->isCustomModule($module_name)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Check if module passes advanced filters that require calculated data.
   */
  protected function passesAdvancedFilters($module_data, $filters) {
    // Unused modules filter - check if module has minimal usage
    if (!empty($filters['unused_only'])) {
      // Consider a module "unused" if it has no fields or views and is not enabled
      $is_potentially_unused = (
        ($module_data['usage_metrics']['fields'] == 0 && $module_data['usage_metrics']['views'] == 0) ||
        ($module_data['status'] !== 'enabled' && empty($module_data['dependents']))
      );
      
      if (!$is_potentially_unused) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Check if module is custom.
   */
  protected function isCustomModule($module_name) {
    $path = $this->getModulePath($module_name);
    return strpos($path, 'modules/custom/') !== FALSE;
  }

  /**
   * Get module path.
   */
  protected function getModulePath($module_name) {
    try {
      return $this->moduleList->getPath($module_name);
    }
    catch (\Exception $e) {
      return 'Unknown';
    }
  }

  /**
   * Get module dependencies.
   */
  protected function getModuleDependencies($module_name, $module_info) {
    $dependencies = [];
    if (!empty($module_info['dependencies'])) {
      foreach ($module_info['dependencies'] as $dependency) {
        // Remove version constraints
        $dep_name = preg_replace('/\s*\([^)]*\)/', '', $dependency);
        $dep_name = str_replace('drupal:', '', $dep_name);
        $dependencies[] = $dep_name;
      }
    }
    return $dependencies;
  }

  /**
   * Get modules that depend on this module.
   */
  protected function getModuleDependents($module_name) {
    $dependents = [];
    $all_modules = $this->moduleList->getAllAvailableInfo();
    
    foreach ($all_modules as $other_module => $info) {
      if (!empty($info['dependencies'])) {
        foreach ($info['dependencies'] as $dependency) {
          $dep_name = preg_replace('/\s*\([^)]*\)/', '', $dependency);
          $dep_name = str_replace('drupal:', '', $dep_name);
          if ($dep_name === $module_name) {
            $dependents[] = $other_module;
            break;
          }
        }
      }
    }
    
    return $dependents;
  }

  /**
   * Get usage metrics for a module.
   */
  protected function getUsageMetrics($module_name) {
    $metrics = [
      'content_types' => 0,
      'fields' => 0,
      'views' => 0,
      'config_items' => 0,
      'field_details' => [],
      'content_type_details' => [],
      'view_details' => [],
    ];

    // Check configuration dependencies
    $configs = $this->configStorage->listAll();
    foreach ($configs as $config_name) {
      $config = $this->configStorage->read($config_name);
      if (!empty($config['dependencies']['module']) && in_array($module_name, $config['dependencies']['module'])) {
        $metrics['config_items']++;

        // Categorize by config type
        if (strpos($config_name, 'node.type.') === 0) {
          $metrics['content_types']++;
          $metrics['content_type_details'][] = str_replace('node.type.', '', $config_name);
        }
        elseif (strpos($config_name, 'field.field.') === 0 || strpos($config_name, 'field.storage.') === 0) {
          $metrics['fields']++;
          $metrics['field_details'][] = $config_name;
        }
        elseif (strpos($config_name, 'views.view.') === 0) {
          $metrics['views']++;
          $metrics['view_details'][] = str_replace('views.view.', '', $config_name);
        }
      }
    }

    return $metrics;
  }

  /**
   * Get configuration data for a module.
   */
  protected function getConfigurationData($module_name) {
    $config_data = [
      'install_configs' => [],
      'optional_configs' => [],
      'schema_files' => [],
    ];

    $module_path = $this->getModulePath($module_name);
    
    // Check for config/install directory
    $install_path = DRUPAL_ROOT . '/' . $module_path . '/config/install';
    if (is_dir($install_path)) {
      $config_data['install_configs'] = array_diff(scandir($install_path), ['.', '..']);
    }

    // Check for config/optional directory
    $optional_path = DRUPAL_ROOT . '/' . $module_path . '/config/optional';
    if (is_dir($optional_path)) {
      $config_data['optional_configs'] = array_diff(scandir($optional_path), ['.', '..']);
    }

    // Check for schema files
    $schema_path = DRUPAL_ROOT . '/' . $module_path . '/config/schema';
    if (is_dir($schema_path)) {
      $config_data['schema_files'] = array_diff(scandir($schema_path), ['.', '..']);
    }

    return $config_data;
  }

  /**
   * Get permissions defined by a module.
   */
  protected function getModulePermissions($module_name) {
    $permissions = [];
    
    try {
      $module_path = $this->getModulePath($module_name);
      $permissions_file = DRUPAL_ROOT . '/' . $module_path . '/' . $module_name . '.permissions.yml';
      
      if (file_exists($permissions_file)) {
        $yaml_content = file_get_contents($permissions_file);
        $parsed_permissions = \Drupal\Component\Serialization\Yaml::decode($yaml_content);
        if (is_array($parsed_permissions)) {
          $permissions = array_keys($parsed_permissions);
        }
      }
    }
    catch (\Exception $e) {
      // Ignore errors
    }

    return $permissions;
  }

  /**
   * Get documentation links for a module.
   */
  protected function getDocumentationLinks($module_name, $module_info) {
    $links = [];
    
    // Check for README file
    $module_path = $this->getModulePath($module_name);
    $readme_files = ['README.md', 'README.txt', 'readme.md', 'readme.txt'];
    foreach ($readme_files as $readme_file) {
      if (file_exists(DRUPAL_ROOT . '/' . $module_path . '/' . $readme_file)) {
        $links['readme'] = '/' . $module_path . '/' . $readme_file;
        break;
      }
    }

    // Check if it's a contrib module and add drupal.org link
    if (!empty($module_info['project']) && $module_info['project'] !== 'drupal') {
      $links['drupal_org'] = 'https://www.drupal.org/project/' . $module_info['project'];
    }

    return $links;
  }


  /**
   * Get CSS class for module status.
   */
  protected function getStatusClass($module_data) {
    $classes = [];
    
    if ($module_data['is_custom']) {
      $classes[] = 'custom-module';
    }
    
    // Check if module is enabled but has minimal usage (no fields or views)
    if ($module_data['status'] === 'enabled' && 
        $module_data['usage_metrics']['fields'] == 0 && 
        $module_data['usage_metrics']['views'] == 0) {
      $classes[] = 'unused-module';
    }
    
    if ($module_data['status'] === 'disabled' && !empty($module_data['dependents'])) {
      $classes[] = 'disabled-with-dependents';
    }
    
    $classes[] = 'status-' . $module_data['status'];
    
    return implode(' ', $classes);
  }

  /**
   * Get all available packages.
   */
  public function getAvailablePackages() {
    $packages = [];
    $all_modules = $this->moduleList->getAllAvailableInfo();
    
    foreach ($all_modules as $module_info) {
      $package = $module_info['package'] ?? 'Other';
      $packages[$package] = $package;
    }
    
    ksort($packages);
    return $packages;
  }

}