<?php

namespace Drupal\system_module_insights\Service;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Service for exporting module data in various formats.
 */
class ModuleExporter {

  /**
   * The module analyzer service.
   *
   * @var \Drupal\system_module_insights\Service\ModuleAnalyzer
   */
  protected $moduleAnalyzer;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a ModuleExporter object.
   */
  public function __construct(
    ModuleAnalyzer $module_analyzer,
    FileSystemInterface $file_system,
    ConfigFactoryInterface $config_factory
  ) {
    $this->moduleAnalyzer = $module_analyzer;
    $this->fileSystem = $file_system;
    $this->configFactory = $config_factory;
  }

  /**
   * Export module data to CSV format.
   *
   * @param array $modules_data
   *   Array of module data.
   * @param array $selected_columns
   *   Array of columns to include in export.
   *
   * @return \Symfony\Component\HttpFoundation\StreamedResponse
   *   The CSV response.
   */
  public function exportToCsv(array $modules_data, array $selected_columns = []) {
    $response = new StreamedResponse();
    $response->headers->set('Content-Type', 'text/csv');
    $response->headers->set('Content-Disposition', 'attachment; filename="module-analysis-' . date('Y-m-d-H-i-s') . '.csv"');

    $response->setCallback(function() use ($modules_data, $selected_columns) {
      $handle = fopen('php://output', 'w');
      
      // Define available columns
      $available_columns = $this->getAvailableColumns();
      
      // Use all columns if none selected
      if (empty($selected_columns)) {
        $selected_columns = array_keys($available_columns);
      }

      // Write header
      $header = [];
      foreach ($selected_columns as $column) {
        $header[] = $available_columns[$column] ?? $column;
      }
      fputcsv($handle, $header);

      // Write data rows
      foreach ($modules_data as $module_data) {
        $row = [];
        foreach ($selected_columns as $column) {
          $row[] = $this->formatCellData($module_data, $column);
        }
        fputcsv($handle, $row);
      }

      fclose($handle);
    });

    return $response;
  }

  /**
   * Export module data to JSON format.
   *
   * @param array $modules_data
   *   Array of module data.
   * @param array $selected_columns
   *   Array of columns to include in export.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The JSON response.
   */
  public function exportToJson(array $modules_data, array $selected_columns = []) {
    // Filter data if specific columns selected
    if (!empty($selected_columns)) {
      $filtered_data = [];
      foreach ($modules_data as $module_name => $module_data) {
        $filtered_module = [];
        foreach ($selected_columns as $column) {
          $filtered_module[$column] = $this->extractColumnValue($module_data, $column);
        }
        $filtered_data[$module_name] = $filtered_module;
      }
      $modules_data = $filtered_data;
    }

    $json_data = [
      'export_timestamp' => date('c'),
      'total_modules' => count($modules_data),
      'modules' => $modules_data,
    ];

    $response = new Response(
      json_encode($json_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
      200,
      [
        'Content-Type' => 'application/json',
        'Content-Disposition' => 'attachment; filename="module-analysis-' . date('Y-m-d-H-i-s') . '.json"',
      ]
    );

    return $response;
  }

  /**
   * Export module data to PDF format.
   *
   * @param array $modules_data
   *   Array of module data.
   * @param array $selected_columns
   *   Array of columns to include in export.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The PDF response.
   */
  public function exportToPdf(array $modules_data, array $selected_columns = []) {
    // For PDF export, we'll create an HTML version and let the browser handle PDF conversion
    // In a production environment, you might want to use a library like TCPDF or wkhtmltopdf
    
    $available_columns = $this->getAvailableColumns();
    if (empty($selected_columns)) {
      $selected_columns = array_keys($available_columns);
    }

    $html = $this->generatePdfHtml($modules_data, $selected_columns, $available_columns);

    $response = new Response(
      $html,
      200,
      [
        'Content-Type' => 'text/html',
        'Content-Disposition' => 'attachment; filename="module-analysis-' . date('Y-m-d-H-i-s') . '.html"',
      ]
    );

    return $response;
  }

  /**
   * Get available export columns.
   *
   * @return array
   *   Array of column keys and labels.
   */
  public function getAvailableColumns() {
    return [
      'name' => 'Module Name',
      'display_name' => 'Display Name',
      'description' => 'Description',
      'package' => 'Package',
      'version' => 'Version',
      'status' => 'Status',
      'dependencies' => 'Dependencies',
      'dependents' => 'Dependents',
      'fields' => 'Fields',
      'views' => 'Views',
      'config_items' => 'Config Items',
      'permissions' => 'Permissions',
      'is_core' => 'Is Core',
      'is_custom' => 'Is Custom',
    ];
  }

  /**
   * Format cell data for export.
   *
   * @param array $module_data
   *   Module data array.
   * @param string $column
   *   Column key.
   *
   * @return string
   *   Formatted cell data.
   */
  protected function formatCellData(array $module_data, $column) {
    switch ($column) {
      case 'dependencies':
      case 'dependents':
        return !empty($module_data[$column]) ? implode(', ', $module_data[$column]) : '';

      case 'permissions':
        return !empty($module_data[$column]) ? implode(', ', $module_data[$column]) : '';

      case 'fields':
        return $module_data['usage_metrics']['fields'] ?? 0;

      case 'views':
        return $module_data['usage_metrics']['views'] ?? 0;

      case 'config_items':
        return $module_data['usage_metrics']['config_items'] ?? 0;

      case 'is_core':
      case 'is_custom':
        return $module_data[$column] ? 'Yes' : 'No';

      default:
        return $module_data[$column] ?? '';
    }
  }

  /**
   * Extract raw column data for export formats that preserve structure.
   *
   * @param array $module_data
   *   Module data array.
   * @param string $column
   *   Column key.
   *
   * @return mixed
   *   Raw column data or NULL when unavailable.
   */
  protected function extractColumnValue(array $module_data, $column) {
    switch ($column) {
      case 'fields':
        return $module_data['usage_metrics']['fields'] ?? 0;

      case 'views':
        return $module_data['usage_metrics']['views'] ?? 0;

      case 'config_items':
        return $module_data['usage_metrics']['config_items'] ?? 0;

      default:
        return $module_data[$column] ?? NULL;
    }
  }

  /**
   * Generate HTML for PDF export.
   *
   * @param array $modules_data
   *   Array of module data.
   * @param array $selected_columns
   *   Selected columns.
   * @param array $available_columns
   *   Available columns.
   *
   * @return string
   *   Generated HTML.
   */
  protected function generatePdfHtml(array $modules_data, array $selected_columns, array $available_columns) {
    $html = '<!DOCTYPE html><html><head>';
    $html .= '<title>Module Analysis Report</title>';
    $html .= '<style>';
    $html .= 'body { font-family: Arial, sans-serif; margin: 20px; }';
    $html .= 'table { width: 100%; border-collapse: collapse; margin-top: 20px; }';
    $html .= 'th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }';
    $html .= 'th { background-color: #f2f2f2; font-weight: bold; }';
    $html .= 'tr:nth-child(even) { background-color: #f9f9f9; }';
    $html .= '.custom-module { background-color: #e8f4fd; }';
    $html .= '.unused-module { background-color: #fff3cd; }';
    $html .= '</style>';
    $html .= '</head><body>';
    
    $html .= '<h1>Module Analysis Report</h1>';
    $html .= '<p>Generated on: ' . date('F j, Y, g:i a') . '</p>';
    $html .= '<p>Total modules: ' . count($modules_data) . '</p>';
    
    $html .= '<table>';
    $html .= '<thead><tr>';
    foreach ($selected_columns as $column) {
      $html .= '<th>' . ($available_columns[$column] ?? $column) . '</th>';
    }
    $html .= '</tr></thead><tbody>';

    foreach ($modules_data as $module_data) {
      $row_class = '';
      if ($module_data['is_custom']) {
        $row_class = 'custom-module';
      }
      elseif ($module_data['status'] === 'enabled' && 
              ($module_data['usage_metrics']['fields'] ?? 0) == 0 && 
              ($module_data['usage_metrics']['views'] ?? 0) == 0) {
        $row_class = 'unused-module';
      }

      $html .= '<tr class="' . $row_class . '">';
      foreach ($selected_columns as $column) {
        $html .= '<td>' . htmlspecialchars($this->formatCellData($module_data, $column)) . '</td>';
      }
      $html .= '</tr>';
    }

    $html .= '</tbody></table>';
    $html .= '</body></html>';

    return $html;
  }

}
