<?php

declare(strict_types=1);

namespace Drupal\tab_title_attention\Form;

use Drupal\Core\Condition\ConditionInterface;
use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Condition\ConditionPluginCollection;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Tab Title Attention settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a \Drupal\tab_title_attention\Form\SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $conditionManager
   *   The condition manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, protected ConditionManager $conditionManager) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.condition'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'tab_title_attention_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['tab_title_attention.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#description' => $this->t("Check to enable the module's functionality."),
      '#default_value' => $this->config('tab_title_attention.settings')->get('enabled'),
    ];
    $form['delay'] = [
      '#type' => 'number',
      '#min' => 0,
      '#step' => 1,
      '#field_suffix' => $this->t('ms'),
      '#title' => $this->t('Animation delay'),
      '#description' => $this->t('The delay before the animation starts, after the tab loses focus.'),
      '#default_value' => $this->config('tab_title_attention.settings')->get('delay'),
    ];
    $form['message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title message'),
      '#description' => $this->t('The message that will appear in the tab title, after it loses focus.'),
      '#default_value' => $this->config('tab_title_attention.settings')->get('message'),
    ];
    $form['animation'] = [
      '#type' => 'radios',
      '#options' => [
        'static' => $this->t('Static'),
        'blink' => $this->t('Blinking (with previous title)'),
        'blink_empty' => $this->t('Blinking (with empty title)'),
        'scroll' => $this->t('Scrolling'),
      ],
      '#required' => TRUE,
      '#title' => $this->t('Animation type'),
      '#description' => $this->t('The type of animation that the title message will play.'),
      '#default_value' => $this->config('tab_title_attention.settings')->get('animation'),
    ];
    $form['animation']['static']['#description'] = $this->t('The message will replace the tab title without any animation.');
    $form['animation']['blink']['#description'] = $this->t('The tab title will blink, switching between the message and the previous tab title.');
    $form['animation']['blink_empty']['#description'] = $this->t('The tab title will blink, switching between the message and an empty title.');
    $form['animation']['scroll']['#description'] = $this->t('The message will appear and scroll through repeatedly from right to left (similar to a marquee).');
    $form['speed'] = [
      '#type' => 'number',
      '#min' => 0,
      '#step' => 1,
      '#field_suffix' => $this->t('ms'),
      '#title' => $this->t('Animation speed'),
      '#description' => $this->t('The speed of the animation (does not apply to "Static").<br><em>Note that some browsers might automatically limit the animation speed for inactive tabs.</em>'),
      '#default_value' => $this->config('tab_title_attention.settings')->get('speed'),
    ];
    $form['conditions'] = [
      '#tree' => TRUE,
    ];
    $form['conditions'] += $this->buildConditions([], $form_state);

    return parent::buildForm($form, $form_state);
  }

  /**
   * Helper function for building the conditions form elements.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form array with the visibility UI added in.
   */
  protected function buildConditions(array $form, FormStateInterface $form_state) {
    $form['conditions_tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Conditions'),
      // @todo This seems to be needed for the group to work as expected, but
      // logically looks strange?
      '#parents' => ['conditions_tabs'],
      '#tree' => TRUE,
    ];

    $conditions = $this->config('tab_title_attention.settings')->get('conditions');
    foreach ($this->conditionManager->getDefinitions() as $condition_id => $definition) {
      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $this->conditionManager->createInstance($condition_id, $conditions[$condition_id] ?? []);
      $form_state->set(['conditions', $condition_id], $condition);
      $condition_form = $condition->buildConfigurationForm([], $form_state);
      $condition_form['#type'] = 'details';
      $condition_form['#title'] = $condition->getPluginDefinition()['label'];
      $condition_form['#group'] = 'conditions_tabs';
      $form[$condition_id] = $condition_form;
    }

    if (isset($form['node_type'])) {
      $form['node_type']['#title'] = $this->t('Content types');
      $form['node_type']['bundles']['#title'] = $this->t('Content types');
      $form['node_type']['negate']['#type'] = 'value';
      $form['node_type']['negate']['#title_display'] = 'invisible';
      $form['node_type']['negate']['#value'] = $form['node_type']['negate']['#default_value'];
    }
    if (isset($form['user_role'])) {
      $form['user_role']['#title'] = $this->t('Roles');
      unset($form['user_role']['roles']['#description']);
      $form['user_role']['negate']['#type'] = 'value';
      $form['user_role']['negate']['#value'] = $form['user_role']['negate']['#default_value'];
    }
    if (isset($form['request_path'])) {
      $form['request_path']['#title'] = $this->t('Pages');
      $form['request_path']['negate']['#type'] = 'radios';
      $form['request_path']['negate']['#default_value'] = (int) $form['request_path']['negate']['#default_value'];
      $form['request_path']['negate']['#title_display'] = 'invisible';
      $form['request_path']['negate']['#options'] = [
        $this->t('Show for the listed pages'),
        $this->t('Hide for the listed pages'),
      ];
    }
    if (isset($form['language'])) {
      $form['language']['negate']['#type'] = 'value';
      $form['language']['negate']['#value'] = $form['language']['negate']['#default_value'];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $this->validateConditions($form, $form_state);
  }

  /**
   * Helper function to independently validate the visibility UI.
   *
   * @param array $form
   *   A nested array form elements comprising the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  protected function validateConditions(array $form, FormStateInterface $form_state) {
    // Validate visibility condition settings.
    foreach ($form_state->getValue('conditions') as $condition_id => $values) {
      // Allow the condition to validate the form.
      $condition = $form_state->get(['conditions', $condition_id]);
      assert($condition instanceof ConditionInterface);
      $condition->validateConfigurationForm($form['conditions'][$condition_id], SubformState::createForSubform($form['conditions'][$condition_id], $form, $form_state));
    }
  }

  /**
   * Helper function to retrieve the conditions configuration form values.
   *
   * @param array $form
   *   A nested array form elements comprising the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  protected function getConditionsConfigurationValue(array $form, FormStateInterface $form_state) {
    $conditionsCollection = new ConditionPluginCollection($this->conditionManager, []);
    foreach ($form_state->getValue('conditions') as $condition_id => $values) {
      // Allow the condition to submit the form.
      $condition = $form_state->get(['conditions', $condition_id]);
      assert($condition instanceof ConditionInterface);
      $condition->submitConfigurationForm($form['conditions'][$condition_id], SubformState::createForSubform($form['conditions'][$condition_id], $form, $form_state));

      $condition_configuration = $condition->getConfiguration();
      $conditionsCollection->addInstanceId($condition_id, $condition_configuration);
    }
    return $conditionsCollection->getConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('tab_title_attention.settings')
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('delay', $form_state->getValue('delay'))
      ->set('message', $form_state->getValue('message'))
      ->set('animation', $form_state->getValue('animation'))
      ->set('speed', $form_state->getValue('speed'))
      ->set('conditions', $this->getConditionsConfigurationValue($form, $form_state))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
