<?php

declare(strict_types=1);

namespace Drupal\Tests\tab_title_attention\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests the JavaScript functionality of the Tab Title Attention module.
 *
 * @group tab_title_attention
 */
final class TabTitleAttentionJavascriptTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['tab_title_attention', 'system'];

  /**
   * A user with authenticated permissions.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user;

  /**
   * A user with admin permissions.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->config('system.site')->set('page.front', '/test-page')->save();
    $this->user = $this->drupalCreateUser([]);
    $this->adminUser = $this->drupalCreateUser([]);
    $this->adminUser->addRole($this->createAdminRole('admin', 'admin'));
    $this->adminUser->save();
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Tests wether the settings are saved and applied via drupalSettings.
   */
  public function testDrupalSettings(): void {
    // Since the animation is disabled by default, check the drupalSettings to
    // make sure that the library is not yet injected.
    $driver = $this->getSession()->getDriver();
    $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
    $this->assertNull($drupalSettings);
    // Change the settings via configFactory.
    \Drupal::configFactory()->getEditable('tab_title_attention.settings')
      ->set('enabled', TRUE)
      ->set('delay', 123)
      ->set('message', 'Test Message!')
      ->set('animation', 'scroll')
      ->set('speed', 456)
      ->save();
    // Check the drupalSettings in JS to see if the settings apply there.
    $this->drupalGet('<front>');
    $settingEnabled = $driver->evaluateScript('drupalSettings.tab_title_attention.enabled');
    $settingDelay = $driver->evaluateScript('drupalSettings.tab_title_attention.delay');
    $settingMessage = $driver->evaluateScript('drupalSettings.tab_title_attention.message');
    $settingAnimation = $driver->evaluateScript('drupalSettings.tab_title_attention.animation');
    $settingSpeed = $driver->evaluateScript('drupalSettings.tab_title_attention.speed');
    $this->assertTrue($settingEnabled);
    $this->assertEquals(123, $settingDelay);
    $this->assertEquals('Test Message!', $settingMessage);
    $this->assertEquals('scroll', $settingAnimation);
    $this->assertEquals(456, $settingSpeed);
  }

  /**
   * Tests the library injection with different permissions.
   */
  public function testPermissions(): void {
    $driver = $this->getSession()->getDriver();
    // Go to the settings page and activate the animation.
    $this->drupalGet('/admin/config/user-interface/tab-title-attention');
    $page = $this->getSession()->getPage();
    $page->checkField('enabled');
    $page->pressButton('edit-submit');
    // Check that, as the admin user, the library is now injected:
    $this->drupalGet('<front>');
    $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
    $this->assertNotNull($drupalSettings);
    // Log in as a regular user without the permission to see the animation and
    // check that the library is not injected anymore.
    $this->drupalLogout();
    $this->drupalLogin($this->user);
    $this->drupalGet('<front>');
    $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
    $this->assertNull($drupalSettings);
    // Now also go to the settings page and see that it returns a 403.
    $this->drupalGet('/admin/config/user-interface/tab-title-attention');
    $session = $this->assertSession();
    $session->pageTextContains('Access denied');
  }

  /**
   * Tests the default conditions.
   */
  public function testDefaultConditions(): void {
    $driver = $this->getSession()->getDriver();
    // First, enable the modules functionality.
    \Drupal::configFactory()->getEditable('tab_title_attention.settings')
      ->set('enabled', TRUE)
      // Negate is true by default.
      ->save();

    // Test the default "Pages" condition:
    // By default conditions, admin pages should be excluded.
    $this->drupalGet('/admin/config');
    $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
    $this->assertNull($drupalSettings);
    $this->drupalGet('<front>');
    $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
    $this->assertNotNull($drupalSettings);
  }

  // phpcs:disable
  // @todo Comment in once https://www.drupal.org/project/drupal/issues/2783897 is solved:
  // /**
  //  * Tests the default conditions inverted.
  //  */
  // public function testDefaultConditionsInverted(): void {
  //   $driver = $this->getSession()->getDriver();
  //   // Invert the conditions and test again vice-versa:
  //   \Drupal::configFactory()->getEditable('tab_title_attention.settings')
  //     ->set('conditions.request_path.negate', FALSE)
  //     ->set('conditions.request_path.pages', "/admin")
  //     ->save();
  // // We now expect tab title attention only on the admin pages:
  //   $this->drupalGet('/admin');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNotNull($drupalSettings);
  //   $this->drupalGet('<front>');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNull($drupalSettings);
  // }
  // @todo Comment in once https://www.drupal.org/project/drupal/issues/2783897 is solved:
  // /**
  //  * Tests with no conditions set.
  //  */
  // public function testEmptyConditions(): void {
  //   $driver = $this->getSession()->getDriver();
  //   // Remove all conditions, so tab_title_attention should appear everywhere:
  //   \Drupal::configFactory()->getEditable('tab_title_attention.settings')
  //     ->set('conditions', [])
  //     ->save();
  //   // We now expect tab title attention only on the admin pages:
  //   $this->drupalGet('/admin/config');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNotNull($drupalSettings);
  //   $this->drupalGet('<front>');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNotNull($drupalSettings);
  // }
  // @todo Comment in once https://www.drupal.org/project/drupal/issues/2783897 is solved:
  // /**
  //  * Tests the response_status conditions.
  //  */
  // public function testResponseStatusConditions(): void {
  //   $driver = $this->getSession()->getDriver();
  //   // Test the "Response Status" condition:
  //   // By default conditions, response status doesn't matter.
  //   $this->drupalGet('/this_page_is_a_404');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNotNull($drupalSettings);
  //   // Set the condition to only show for 200 status codes and check again.
  //   \Drupal::configFactory()->getEditable('tab_title_attention.settings')
  //   // Remove the default request_path condition:
  //     ->clear('conditions.request_path')
  //     // Set the response status condition:
  //     ->set('conditions.response_status.status_codes.404', TRUE)
  //     ->save();
  //   $this->drupalGet('<front>');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNull($drupalSettings);
  //   $this->drupalGet('/this_page_is_a_404');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNotNull($drupalSettings);
  // }
  // @todo Comment in once https://www.drupal.org/project/drupal/issues/2783897 is solved:
  // /**
  //  * Tests the roles conditions.
  //  */
  // public function testRolesConditions(): void {
  //   $driver = $this->getSession()->getDriver();
  //   // Test the "Roles" condition:
  //   // By default conditions, role doesn't matter.
  //   $this->drupalGet('<front>');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNotNull($drupalSettings);
  //   // Set the condition to only show for admins and check again.
  //   \Drupal::configFactory()->getEditable('tab_title_attention.settings')
  //   // Remove the default request_path condition:
  //     ->clear('conditions.request_path')
  //     // Set the roles conditions:
  //     ->set('conditions.user_role.roles.administrator', TRUE)
  //     ->save();
  //   $this->drupalGet('<front>');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNotNull($drupalSettings);
  //   $this->drupalLogout();
  //   $this->drupalLogin($this->user);
  //   $this->drupalGet('<front>');
  //   $drupalSettings = $driver->evaluateScript('drupalSettings.tab_title_attention');
  //   $this->assertNull($drupalSettings);
  //   // Reset conditions.
  //   \Drupal::configFactory()->getEditable('tab_title_attention.settings')
  //     ->set('conditions.user_role.roles.administrator', FALSE)
  //     ->save();
  //   $this->drupalLogout();
  //   $this->drupalLogin($this->adminUser);
  // }
  // phpcs:enable
}
