<?php

declare(strict_types=1);

namespace Drupal\tabby_viewfield\Plugin\Field\FieldFormatter;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\viewfield\Plugin\Field\FieldFormatter\ViewfieldFormatterDefault;

/**
 * Plugin implementation of the 'Tabby Tabs' formatter.
 */
#[FieldFormatter(
  id: 'tabby_viewfield_tabby_tabs',
  label: new TranslatableMarkup('Tabby Tabs'),
  field_types: ['viewfield'],
)]
class TabbyTabsFormatter extends ViewfieldFormatterDefault {
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    $settings = parent::defaultSettings();
    $settings['tab_titles'] = '';
    $settings['tab_behaviors'] = '';
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $form = parent::settingsForm($form, $form_state);

    $form['tab_titles'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tab titles'),
      '#default_value' => $this->getSetting('tab_titles'),
      '#description' => $this->t('Comma separated list of tab titles.'),
    ];

    $form['tab_behaviors'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tab behaviors'),
      '#default_value' => $this->getSetting('tab_behaviors'),
      '#description' => $this->t('JSON array of array of tab behaviors. E.g. [["masonry"],["sliderInit"]]'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $settings = $this->getSettings();
    $summary = parent::settingsSummary();
    $summary[] = $this->t('Tab titles: @tab_titles', [
      '@tab_titles' => $settings['tab_titles'],
    ]);
    if ($settings['tab_behaviors']) {
      $summary[] = $this->t('Tab behaviors: @tab_behaviors', [
        '@tab_behaviors' => $settings['tab_behaviors'],
      ]);
    }
    return $summary;
  }

  /**
   * {@inheritDoc}
   */
  public function view(FieldItemListInterface $items, $langcode = NULL): array {
    $settings = $this->getSettings();
    $tabTitles = explode(',', $settings['tab_titles']);
    try {
      $behaviors = Json::decode($settings['tab_behaviors']);
    }
    catch (\Throwable) {
      $behaviors = [];
    }

    $elements = parent::view($items, $langcode);
    $keys = Element::children($elements);

    $labels = [];
    $content = [];

    foreach ($keys as $key) {
      // Pass custom tab title though t to apply any translations.
      $labels[] = empty($tabTitles[$key])
        ? 'Tab ' . ($key + 1)
        : $this->t($tabTitles[$key]);
      $content[] = $elements[$key];
    }

    return [
      '#theme' => 'tabby_tabs',
      '#labels' => $labels,
      '#content' => $content,
      '#behaviors' => $behaviors,
    ];

  }

}
