<?php

namespace Drupal\Tests\tabby_viewfield\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\user\UserInterface;

/**
 * Tests the field widget.
 *
 * @group tabby_viewfield
 */
class TabbyViewfieldFormatterTest extends WebDriverTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'views',
    'field_ui',
    'viewfield',
    'tabby',
    'tabby_viewfield',
    'tabby_viewfield_test',
  ];

  /**
   * A user with permission to manage and use the field.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->drupalCreateContentType([
      'type' => 'page',
      'name' => 'Basic page',
      'display_submitted' => FALSE,
    ]);
    $this->drupalCreateNode([
      'type' => 'page',
      'title' => 'Page 1',
    ]);
    $this->adminUser = $this->drupalCreateUser([
      'access content',
      'bypass node access',
      'administer nodes',
      'administer content types',
      'administer node fields',
      'administer node display',
    ]);
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Tests the widget.
   */
  public function testFormatter() {

    // Create the viewfield field.
    $this->drupalGet('admin/structure/types/manage/page/fields/add-field/viewfield/false', ['query' => ['entity_type' => 'node']]);
    $fieldLabel = 'View field';
    $this->submitForm(['label' => $fieldLabel], 'Continue');

    // Configure the field.
    $this->getSession()->getPage()->fillField('field_storage[subform][cardinality_number]', '2');
    $this->assertSession()->assertWaitOnAjaxRequest();

    $this->getSession()->getPage()->checkField('settings[force_default]');
    $this->getSession()->getPage()->checkField('settings[allowed_views][test_one]');
    $this->getSession()->getPage()->checkField('settings[allowed_views][test_two]');
    $this->getSession()->getPage()->checkField('set_default_value');
    $this->getSession()->getPage()->selectFieldOption('default_value_input[field_view_field][0][target_id]', 'test_one');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->getSession()->getPage()->selectFieldOption('default_value_input[field_view_field][1][target_id]', 'test_two');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->submitForm([], 'Save');

    $this->assertSession()->waitForElement('css', '.messages--status');

    $this->htmlOutput($this->getSession()->getPage()->getHtml());

    $this->assertSession()->pageTextContains("Saved $fieldLabel configuration.");

    $this->htmlOutput('After press save.');

    // Change form display to this widget.
    $this->drupalGet('admin/structure/types/manage/page/display');

    $this->getSession()->getPage()->findButton('Show row weights')->click();

    $this->getSession()->getPage()->selectFieldOption('fields[field_view_field][region]', 'content');
    $this->getSession()->getPage()->selectFieldOption('fields[field_view_field][type]', 'tabby_viewfield_tabby_tabs');
    $this->assertSession()->assertWaitOnAjaxRequest();

    $this->submitForm([], 'Save');

    // Set expected values.
    $expectedTabOne = 'Tab 1';
    $expectedTabTwo = 'Tab 2';
    $expectedTabOneContents = 'Test view one';
    $expectedTabTwoContents = 'Test view two';

    // Set custom tab titles.
    $expectedTabOne = 'Test Tab 1';
    $expectedTabTwo = 'Test Tab 2';
    $this->drupalGet('admin/structure/types/manage/page/display');
    $this->getSession()->getPage()->pressButton('field_view_field_settings_edit');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->getSession()->getPage()->fillField('fields[field_view_field][settings_edit_form][settings][tab_titles]', "$expectedTabOne,$expectedTabTwo");
    $this->submitForm([], 'Update');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $this->submitForm([], 'Save');

    // Test the formatter.
    $this->drupalGet('node/1');

    $this->getSession()->getPage()->hasContent($expectedTabOne);
    $this->getSession()->getPage()->hasContent($expectedTabTwo);

    // Only first tab should be visible.
    $tabOne = $this->getSession()->getPage()->find('css', '#tabby-1-0');
    $this->assertNotNull($tabOne);
    $tabOneContents = $tabOne->getText();

    $tabTwo = $this->getSession()->getPage()->find('css', '#tabby-1-1');
    $this->assertNotNull($tabTwo);
    $tabTwoContents = $tabTwo->getText();

    $this->assertEquals($expectedTabOneContents, $tabOneContents, 'First tab contents should be visible.');
    $this->assertEquals('', $tabTwoContents, 'Second tab contents should be hidden.');

    // Second tab should be visible.
    $this->getSession()->getPage()->clickLink($expectedTabTwo);

    $tabOneContents = $this->getSession()->getPage()->find('css', '#tabby-1-0')->getText();
    $tabTwoContents = $this->getSession()->getPage()->find('css', '#tabby-1-1')->getText();

    $this->assertEquals('', $tabOneContents);
    $this->assertEquals($expectedTabTwoContents, $tabTwoContents);
  }

}
