<?php

declare(strict_types=1);

namespace Drupal\table_header_scope_attribute\Plugin\Filter;

use Drupal\Component\Utility\Html;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filter\Attribute\Filter;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\Plugin\FilterInterface;

/**
 * Provides a filter to set scope attribute for table headers.
 */
#[Filter(
  id: 'table_header_scope_attribute',
  title: new TranslatableMarkup("Set scope attribute for table headers"),
  type: FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
  description: new TranslatableMarkup("Applies the <code>scope</code> attribute on <code>&lt;th&gt;</code> elements."),
)]
class TableHeaderScopeAttribute extends FilterBase {

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode): FilterProcessResult {
    $result = new FilterProcessResult($text);

    // Scope attributes can only be set when there are <th> elements.
    if (stripos($text, '<th') !== FALSE) {
      $dom = Html::load($text);
      $xpath = new \DOMXPath($dom);

      // Process each table separately.
      foreach ($xpath->query('//table') as $table) {
        // There should be at least one <td> element in the table; no need to
        // set scope attribute if there are only <th> elements.
        if ($xpath->query('.//td', $table)->count() > 0) {
          // Loop through all rows of this table.
          foreach ($xpath->query('.//tr', $table) as $row) {
            // The header elements belong to the column, if there are only <th>
            // elements in this row (i.e., no <td> elements). Otherwise, they
            // belong to the row.
            $belongs_to = $xpath->query('.//td', $row)->count() > 0 ? 'row' : 'col';

            // Set the scope attribute on each header element.
            /** @var \DOMElement $header_element */
            foreach ($xpath->query('.//th', $row) as $header_element) {
              // In case the header element spans multiple columns or rows, set
              // the scope attribute to "colgroup" or "rowgroup" respectively.
              $scope_value = $header_element->getAttribute($belongs_to . 'span') > 1 ? $belongs_to . 'group' : $belongs_to;

              $header_element->setAttribute('scope', $scope_value);
            }
          }
        }
      }

      // Finally, serialize the DOM back to text.
      $result->setProcessedText(Html::serialize($dom));
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE): string|null {
    return (string) $this->t('Automatically sets the <code>scope</code> attribute for <code>th</code> elements, such as <code>&lt;th scope="col"&gt;&lt;/th&gt;</code>.');
  }

}
