<?php

declare(strict_types=1);

namespace Drupal\table_header_scope_attribute\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Defines hooks provided by the Table Header Scope Attribute module.
 */
class TableHeaderScopeAttributeHooks {

  use StringTranslationTrait;

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(string $route_name, RouteMatchInterface $route_match): ?string {
    if ($route_name === 'help.page.table_header_scope_attribute') {
      $output = '<h2>' . $this->t('About') . '</h2>';
      $output .= '<p>' . $this->t('The Table Header Scope Attribute module provides a filter that adds scope attributes to table header cells (&lt;th&gt; elements) in user-submitted content. This improves accessibility by helping screen readers understand the relationship between header cells and data cells in tables. For more information, see the <a href=":project_link">online documentation for the Table Header Scope Attribute module</a>.', [
        ':project_link' => 'https://www.drupal.org/project/table_header_scope_attribute',
      ]) . '</p>';
      $output .= '<h2>' . $this->t('Uses') . '</h2>';
      $output .= '<dl>';
      $output .= '<dt>' . $this->t('Enabling the filter') . '</dt>';
      $output .= '<dd>' . $this->t('To use the filter, navigate to the <a href=":formats">Text formats page</a>, edit the desired text format, and enable the "Add scope attribute to table headers" filter. Adjust the filter\'s weight to ensure it runs in the correct order relative to other filters.', [
        ':formats' => Url::fromRoute('filter.admin_overview')->toString(),
      ]) . '</dd>';
      $output .= '<dt>' . $this->t('Configuring the filter') . '</dt>';
      $output .= '<dd>' . $this->t('No additional configuration is necessary for this filter. Once enabled, it will automatically add scope attributes to &lt;th&gt; elements in user-submitted content.') . '</dd>';
      $output .= '</dl>';
      return $output;
    }

    return NULL;
  }

  /**
   * Implements hook_form_FORM_ID_alter() for filter_format_add_form and filter_format_edit_form.
   */
  #[Hook('form_filter_format_add_form_alter')]
  #[Hook('form_filter_format_edit_form_alter')]
  public function formFilterFormatAddEditAlter(array &$form, FormStateInterface $form_state): void {
    $form['#validate'][] = [$this, 'validateFilterFormatAddEditForm'];
  }

  /**
   * Validates the filter format add/edit form.
   *
   * When both 'table_header_scope_attribute' and
   * 'table_header_scope_attribute_empty_th_to_td' filters are enabled, the
   * 'table_header_scope_attribute' filter must run before the
   * 'table_header_scope_attribute_empty_th_to_td' filter (otherwise, the role
   *  attribute is set incorrectly as it is based on <td> elements).
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateFilterFormatAddEditForm(array &$form, FormStateInterface $form_state): void {
    $filters = $form_state->getValue('filters');

    $scope_filter_enabled = !empty($filters['table_header_scope_attribute']['status']);
    $empty_th_filter_enabled = !empty($filters['table_header_scope_attribute_empty_th_to_td']['status']);

    if ($scope_filter_enabled && $empty_th_filter_enabled) {
      $scope_filter_weight = (int) $filters['table_header_scope_attribute']['weight'];
      $empty_th_filter_weight = (int) $filters['table_header_scope_attribute_empty_th_to_td']['weight'];

      if ($empty_th_filter_weight <= $scope_filter_weight) {
        $form_state->setError(
          $form['filters']['order'],
          $this->t('The "Convert empty <th> to <td> and add scope attribute" filter must run after the "Add scope attribute to table headers" filter. Please adjust the weights accordingly.'),
        );
      }
    }
  }

}
