<?php

declare(strict_types=1);

namespace Drupal\table_header_scope_attribute\Plugin\Filter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filter\Attribute\Filter;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\Plugin\FilterInterface;
use Drupal\table_header_scope_attribute\HtmlElementValidatorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a filter to transform empty <th> into <td>.
 */
#[Filter(
  id: 'table_header_scope_attribute_empty_th_to_td',
  title: new TranslatableMarkup('Transform empty table header to table data'),
  type: FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
  description: new TranslatableMarkup('Transforms empty <code>&lt;th&gt;</code> elements into <code>&lt;td&gt;</code> elements.'),
)]
class EmptyTableHeaderToTableData extends FilterBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs an EmptyTableHeaderToTableData object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\table_header_scope_attribute\HtmlElementValidatorInterface $htmlElementValidator
   *   The HTML element validator service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected HtmlElementValidatorInterface $htmlElementValidator,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('table_header_scope_attribute.html_element_validator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode): FilterProcessResult {
    $result = new FilterProcessResult($text);

    // Header elements can only be transformed when there are <th> elements.
    if (stripos($text, '<th') === FALSE) {
      return $result;
    }

    $dom = Html::load($text);
    $xpath = new \DOMXPath($dom);

    // Transform each empty <th> element into a <td> element.
    foreach ($xpath->query('//table//th') as $header_element) {
      if ($this->htmlElementValidator->isElementContentEmpty($header_element)) {
        // Create a new <td> element.
        $td_element = $dom->createElement('td');

        // Copy all child nodes from <th> to <td>.
        foreach ($header_element->childNodes as $child) {
          $td_element->appendChild($child->cloneNode(TRUE));
        }

        // Copy attributes from <th> to <td>, excluding scope which is invalid
        // for <td>.
        foreach ($header_element->attributes as $attr) {
          if ($attr->nodeName !== 'scope') {
            $td_element->setAttribute($attr->nodeName, $attr->nodeValue);
          }
        }

        // Replace <th> with <td>.
        $header_element->parentNode->replaceChild($td_element, $header_element);
      }
    }

    // Save the modified HTML back to the result.
    $result->setProcessedText(Html::serialize($dom));

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE): string|null {
    return (string) $this->t('Transforms empty <code>&lt;th&gt;</code> elements into <code>&lt;td&gt;</code> elements.');
  }

}
