<?php

declare(strict_types=1);

namespace Drupal\Tests\table_header_scope_attribute\Kernel;

use Drupal\filter\Entity\FilterFormat;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the combined behavior of both filter plugins.
 *
 * These integration tests verify that both filters work correctly together,
 * including scenarios where they're configured in the wrong order. While form
 * validation prevents incorrect ordering during manual configuration, filters
 * can still be misconfigured through config imports, drush commands, or direct
 * YAML editing. Testing both correct and incorrect order serves as regression
 * protection and documents the expected (albeit incorrect) behavior.
 */
#[Group('table_header_scope_attribute')]
class CombinedFiltersTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'filter',
    'table_header_scope_attribute',
  ];

  /**
   * Tests that both filters work correctly when enabled together.
   */
  #[DataProvider('combinedFiltersProvider')]
  public function testCombinedFilters(string $input, string $expected, bool $correct_order): void {
    // Create a filter format with both filters enabled.
    $filter_format = FilterFormat::create([
      'format' => 'test_format',
      'name' => 'Test format',
      'filters' => [
        'table_header_scope_attribute' => [
          'status' => TRUE,
          'weight' => $correct_order ? 0 : 10,
        ],
        'table_header_scope_attribute_empty_th_to_td' => [
          'status' => TRUE,
          'weight' => $correct_order ? 10 : 0,
        ],
      ],
    ]);
    $filter_format->save();

    // Process the input through the filter format.
    $processed = (string) check_markup($input, 'test_format');

    $this->assertSame($expected, $processed);
  }

  /**
   * Provides test data for combined filter processing.
   *
   * @return \Generator
   *   Test cases with input HTML, expected output, and filter order.
   */
  public static function combinedFiltersProvider(): \Generator {
    yield 'table with empty header in first column, correct order' => [
      '<table><thead><tr><th></th><th>Column 1</th></tr></thead><tbody><tr><th>Row 1</th><td>Data</td></tr></tbody></table>',
      '<table><thead><tr><td></td><th scope="col">Column 1</th></tr></thead><tbody><tr><th scope="row">Row 1</th><td>Data</td></tr></tbody></table>',
      TRUE,
    ];

    yield 'table with multiple empty headers, correct order' => [
      '<table><thead><tr><th></th><th>Col 1</th><th></th></tr></thead><tbody><tr><td>Data 1</td><td>Data 2</td><td>Data 3</td></tr></tbody></table>',
      '<table><thead><tr><td></td><th scope="col">Col 1</th><td></td></tr></thead><tbody><tr><td>Data 1</td><td>Data 2</td><td>Data 3</td></tr></tbody></table>',
      TRUE,
    ];

    yield 'table with empty row headers, correct order' => [
      '<table><tbody><tr><th></th><td>Data 1</td></tr><tr><th>Row 2</th><td>Data 2</td></tr></tbody></table>',
      '<table><tbody><tr><td></td><td>Data 1</td></tr><tr><th scope="row">Row 2</th><td>Data 2</td></tr></tbody></table>',
      TRUE,
    ];

    yield 'complex table with mixed empty headers, correct order' => [
      '<table><thead><tr><th></th><th colspan="2">Group</th></tr><tr><th></th><th>Col 1</th><th>Col 2</th></tr></thead><tbody><tr><th>Row 1</th><td>A</td><td>B</td></tr></tbody></table>',
      '<table><thead><tr><td></td><th colspan="2" scope="colgroup">Group</th></tr><tr><td></td><th scope="col">Col 1</th><th scope="col">Col 2</th></tr></thead><tbody><tr><th scope="row">Row 1</th><td>A</td><td>B</td></tr></tbody></table>',
      TRUE,
    ];

    yield 'table with empty header in first column, incorrect order' => [
      '<table><thead><tr><th></th><th>Column 1</th></tr></thead><tbody><tr><th>Row 1</th><td>Data</td></tr></tbody></table>',
      // When empty-th runs first, it converts empty <th> to <td> BEFORE the
      // scope attribute is applied. Then scope filter sees only <th> in rows
      // with <td>, so it applies scope="row" to ALL remaining <th>.
      '<table><thead><tr><td></td><th scope="row">Column 1</th></tr></thead><tbody><tr><th scope="row">Row 1</th><td>Data</td></tr></tbody></table>',
      FALSE,
    ];

    yield 'complex table with mixed empty headers, incorrect order' => [
      '<table><thead><tr><th></th><th colspan="2">Group</th></tr><tr><th></th><th>Col 1</th><th>Col 2</th></tr></thead><tbody><tr><th>Row 1</th><td>A</td><td>B</td></tr></tbody></table>',
      // Incorrect: all remaining <th> get scope="row" because empty ones were
      // converted first. This test case demonstrates the undesirable
      // behavior that results from misconfigured filter order.
      '<table><thead><tr><td></td><th colspan="2" scope="row">Group</th></tr><tr><td></td><th scope="row">Col 1</th><th scope="row">Col 2</th></tr></thead><tbody><tr><th scope="row">Row 1</th><td>A</td><td>B</td></tr></tbody></table>',
      FALSE,
    ];
  }

}
