<?php

declare(strict_types=1);

namespace Drupal\Tests\table_header_scope_attribute\Unit\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\table_header_scope_attribute\Hook\TableHeaderScopeAttributeHooks;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the validation logic in TableHeaderScopeAttributeHooks.
 */
#[Group('table_header_scope_attribute')]
class TableHeaderScopeAttributeHooksTest extends UnitTestCase {

  /**
   * The form state mock.
   */
  private FormStateInterface&MockObject $formState;

  /**
   * The TableHeaderScopeAttributeHooks instance.
   */
  private TableHeaderScopeAttributeHooks $hooks;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->formState = $this->createMock(FormStateInterface::class);

    $this->hooks = new TableHeaderScopeAttributeHooks();
    $this->hooks->setStringTranslation($this->getStringTranslationStub());
  }

  /**
   * Ensures an error is raised when the filters are not correctly ordered.
   *
   * When both filters are enabled, 'table_header_scope_attribute' must run
   * before 'table_header_scope_attribute_empty_th_to_td'. Otherwise, empty <th>
   * elements are converted to <td>, which results in incorrect scope
   * attributes (i.e., only scope="row").
   *
   * @param array $filters
   *   The filters' configuration.
   * @param bool $expected_error
   *   Whether an error should be set.
   */
  #[DataProvider('provideFilterConfigurations')]
  public function testValidateFilterFormatAddEditForm(array $filters, bool $expected_error): void {
    $form = [
      'filters' => [
        'order' => [],
      ],
    ];

    // Set up the form state mock to return the provided filters.
    $this->formState->expects($this->once())
      ->method('getValue')
      ->with('filters')
      ->willReturn($filters);

    // Set expectations based on whether an error is expected.
    if ($expected_error) {
      $this->formState->expects($this->once())
        ->method('setError')
        ->with(
          $form['filters']['order'],
          $this->isInstanceOf(TranslatableMarkup::class)
        );
    }
    else {
      $this->formState->expects($this->never())
        ->method('setError');
    }

    $this->hooks->validateFilterFormatAddEditForm($form, $this->formState);
  }

  /**
   * Data provider for testValidateFilterFormatAddEditForm().
   *
   * @return \Generator
   *   Test cases with filters configuration and expected error state.
   */
  public static function provideFilterConfigurations(): \Generator {
    yield 'both filters disabled' => [
      [
        'table_header_scope_attribute' => [
          'status' => FALSE,
        ],
        'table_header_scope_attribute_empty_th_to_td' => [
          'status' => FALSE,
        ],
      ],
      FALSE,
    ];

    yield 'only scope filter enabled' => [
      [
        'table_header_scope_attribute' => [
          'status' => TRUE,
        ],
        'table_header_scope_attribute_empty_th_to_td' => [
          'status' => FALSE,
        ],
      ],
      FALSE,
    ];

    yield 'only empty th filter enabled' => [
      [
        'table_header_scope_attribute' => [
          'status' => FALSE,
        ],
        'table_header_scope_attribute_empty_th_to_td' => [
          'status' => TRUE,
        ],
      ],
      FALSE,
    ];

    yield 'both enabled, correct order (scope before empty th)' => [
      [
        'table_header_scope_attribute' => [
          'status' => TRUE,
          'weight' => 0,
        ],
        'table_header_scope_attribute_empty_th_to_td' => [
          'status' => TRUE,
          'weight' => 10,
        ],
      ],
      FALSE,
    ];

    yield 'both enabled, incorrect order (empty th before scope)' => [
      [
        'table_header_scope_attribute' => [
          'status' => TRUE,
          'weight' => 10,
        ],
        'table_header_scope_attribute_empty_th_to_td' => [
          'status' => TRUE,
          'weight' => 0,
        ],
      ],
      TRUE,
    ];

    yield 'both enabled, same weight (error)' => [
      [
        'table_header_scope_attribute' => [
          'status' => TRUE,
          'weight' => 5,
        ],
        'table_header_scope_attribute_empty_th_to_td' => [
          'status' => TRUE,
          'weight' => 5,
        ],
      ],
      TRUE,
    ];
  }

}
