<?php

declare(strict_types=1);

namespace Drupal\Tests\table_header_scope_attribute\Unit;

use Drupal\Component\Utility\Html;
use Drupal\table_header_scope_attribute\HtmlElementValidator;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the HTML element validator service.
 */
#[CoversClass(HtmlElementValidator::class)]
#[Group('table_header_scope_attribute')]
class HtmlElementValidatorTest extends UnitTestCase {

  /**
   * The HTML element validator service.
   */
  protected HtmlElementValidator $validator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->validator = new HtmlElementValidator();
  }

  /**
   * Tests the isElementContentEmpty method.
   */
  #[DataProvider('isElementContentEmptyProvider')]
  public function testIsElementContentEmpty(string $html, string $tag_name, bool $expected): void {
    $dom = Html::load($html);
    $element = $dom->getElementsByTagName($tag_name)->item(0);

    // Ensure we have a DOMElement.
    $this->assertInstanceOf(\DOMElement::class, $element);

    $result = $this->validator->isElementContentEmpty($element);

    $this->assertSame($expected, $result);
  }

  /**
   * Provides data to test the isElementContentEmpty method.
   *
   * @return \Generator
   *   Yields test data arrays containing:
   *     - html: The HTML string containing an element to test.
   *     - tag_name: The tag name of the element to test.
   *     - expected: The expected result (TRUE if empty, FALSE otherwise).
   */
  public static function isElementContentEmptyProvider(): \Generator {
    // Test with <th> elements (original use case).
    yield 'th: completely empty header' => [
      '<th></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with only spaces' => [
      '<th>   </th>',
      'th',
      TRUE,
    ];

    yield 'th: header with only tabs' => [
      '<th>		</th>',
      'th',
      TRUE,
    ];

    yield 'th: header with only newlines' => [
      '<th>

</th>',
      'th',
      TRUE,
    ];

    yield 'th: header with non-breaking space' => [
      '<th>&nbsp;</th>',
      'th',
      TRUE,
    ];

    yield 'th: header with multiple non-breaking spaces' => [
      '<th>&nbsp;&nbsp;&nbsp;</th>',
      'th',
      TRUE,
    ];

    yield 'th: header with mixed whitespace' => [
      '<th>  &nbsp; 	</th>',
      'th',
      TRUE,
    ];

    yield 'th: header with zero-width space' => [
      '<th>&#8203;</th>',
      'th',
      TRUE,
    ];

    yield 'th: header with comment' => [
      '<th><!-- This is a comment --></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with text' => [
      '<th>Header</th>',
      'th',
      FALSE,
    ];

    yield 'th: header with text and whitespace' => [
      '<th>  Header  </th>',
      'th',
      FALSE,
    ];

    yield 'th: header with number' => [
      '<th>1</th>',
      'th',
      FALSE,
    ];

    yield 'th: header with special character' => [
      '<th>#</th>',
      'th',
      FALSE,
    ];

    yield 'th: header with HTML entity' => [
      '<th>&amp;</th>',
      'th',
      FALSE,
    ];

    yield 'th: header with multiple text nodes' => [
      '<th>Text1&nbsp;Text2</th>',
      'th',
      FALSE,
    ];

    yield 'th: header with nested element' => [
      '<th><span>Text</span></th>',
      'th',
      FALSE,
    ];

    yield 'th: header with empty nested element' => [
      '<th><span></span></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with span containing only whitespace' => [
      '<th><span> </span></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with span containing only non-breaking space' => [
      '<th><span>&nbsp;</span></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with multiple empty spans' => [
      '<th><span></span><span> </span></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with nested empty spans' => [
      '<th><span><span></span></span></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with span containing text' => [
      '<th><span>Text</span></th>',
      'th',
      FALSE,
    ];

    yield 'th: header with image' => [
      '<th><img src="test.jpg" alt="Test"></th>',
      'th',
      FALSE,
    ];

    yield 'th: header with image inside span' => [
      '<th><span><img src="test.jpg" alt="Test"></span></th>',
      'th',
      FALSE,
    ];

    yield 'th: header with empty span and whitespace' => [
      '<th> <span></span> </th>',
      'th',
      TRUE,
    ];

    // Test with <td> elements.
    yield 'td: empty cell' => [
      '<td></td>',
      'td',
      TRUE,
    ];

    yield 'td: cell with text' => [
      '<td>Content</td>',
      'td',
      FALSE,
    ];

    yield 'td: cell with image' => [
      '<td><img src="icon.png" alt="Icon"></td>',
      'td',
      FALSE,
    ];

    yield 'td: cell with empty span' => [
      '<td><span></span></td>',
      'td',
      TRUE,
    ];

    // Test with <div> elements.
    yield 'div: empty div' => [
      '<div></div>',
      'div',
      TRUE,
    ];

    yield 'div: div with whitespace' => [
      '<div>   </div>',
      'div',
      TRUE,
    ];

    yield 'div: div with text' => [
      '<div>Text content</div>',
      'div',
      FALSE,
    ];

    yield 'div: div with nested empty spans' => [
      '<div><span><span></span></span></div>',
      'div',
      TRUE,
    ];

    yield 'div: div with image in nested span' => [
      '<div><span><img src="photo.jpg" alt="Photo"></span></div>',
      'div',
      FALSE,
    ];

    // Test with <span> elements.
    yield 'span: empty span' => [
      '<span></span>',
      'span',
      TRUE,
    ];

    yield 'span: span with text' => [
      '<span>Text</span>',
      'span',
      FALSE,
    ];

    yield 'span: span with nested image' => [
      '<span><img src="test.jpg" alt="Test"></span>',
      'span',
      FALSE,
    ];

    // Test with <p> elements.
    yield 'p: empty paragraph' => [
      '<p></p>',
      'p',
      TRUE,
    ];

    yield 'p: paragraph with only non-breaking spaces' => [
      '<p>&nbsp;&nbsp;</p>',
      'p',
      TRUE,
    ];

    yield 'p: paragraph with text' => [
      '<p>This is a paragraph.</p>',
      'p',
      FALSE,
    ];

    yield 'p: paragraph with br tag' => [
      '<p><br></p>',
      'p',
      FALSE,
    ];

    // Test with various void elements.
    yield 'div: div with hr' => [
      '<div><hr></div>',
      'div',
      FALSE,
    ];

    yield 'div: div with input' => [
      '<div><input type="text"></div>',
      'div',
      FALSE,
    ];

    yield 'div: div with multiple void elements' => [
      '<div><br><hr></div>',
      'div',
      FALSE,
    ];

    // Additional edge cases.
    yield 'th: header with deeply nested empty spans' => [
      '<th><span><span><span></span></span></span></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with mixed empty and whitespace spans' => [
      '<th><span> </span><span>&nbsp;</span><span></span></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with link containing text' => [
      '<th><a href="#">Link</a></th>',
      'th',
      FALSE,
    ];

    yield 'th: header with empty link' => [
      '<th><a href="#"></a></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with link containing image' => [
      '<th><a href="#"><img src="icon.png" alt="Icon"></a></th>',
      'th',
      FALSE,
    ];

    yield 'th: header with strong tag containing text' => [
      '<th><strong>Bold</strong></th>',
      'th',
      FALSE,
    ];

    yield 'th: header with empty strong tag' => [
      '<th><strong></strong></th>',
      'th',
      TRUE,
    ];

    yield 'div: div with link element (not void)' => [
      '<div><link rel="stylesheet" href="style.css"></div>',
      'div',
      FALSE,
    ];

    yield 'th: header with Unicode whitespace characters' => [
      '<th>&#x200B;&#x200C;&#x200D;&#xFEFF;</th>',
      'th',
      TRUE,
    ];

    yield 'th: header with tab character entity' => [
      '<th>&#9;</th>',
      'th',
      TRUE,
    ];

    yield 'span: span with only comment' => [
      '<span><!-- comment --></span>',
      'span',
      TRUE,
    ];

    yield 'div: div with CDATA section' => [
      '<div><![CDATA[]]></div>',
      'div',
      TRUE,
    ];

    // Test attribute handling: attributes should not be considered content.
    yield 'th: header with id attribute' => [
      '<th id="col-header"></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with empty id attribute' => [
      '<th id=""></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with class attribute' => [
      '<th class="header-cell"></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with title attribute' => [
      '<th title="Tooltip text"></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with data attribute' => [
      '<th data-info="metadata"></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with multiple attributes' => [
      '<th id="header-1" class="sortable" data-sort="asc"></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with scope attribute' => [
      '<th scope="col"></th>',
      'th',
      TRUE,
    ];

    yield 'th: header with attributes and whitespace' => [
      '<th id="test">   </th>',
      'th',
      TRUE,
    ];

    yield 'th: header with attributes and empty child element' => [
      '<th class="empty"><span></span></th>',
      'th',
      TRUE,
    ];

    yield 'td: cell with colspan attribute' => [
      '<td colspan="2"></td>',
      'td',
      TRUE,
    ];

    yield 'div: div with style attribute' => [
      '<div style="color: red;"></div>',
      'div',
      TRUE,
    ];

    yield 'span: span with onclick attribute' => [
      '<span onclick="alert();"></span>',
      'span',
      TRUE,
    ];
  }

}
