<?php

declare(strict_types=1);

namespace Drupal\Tests\table_header_scope_attribute\Unit\Plugin;

use Drupal\table_header_scope_attribute\HtmlElementValidatorInterface;
use Drupal\table_header_scope_attribute\Plugin\Filter\EmptyTableHeaderToTableData;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\Stub;

/**
 * Tests the text filter to transform empty <th> into <td>.
 */
#[CoversClass(EmptyTableHeaderToTableData::class)]
#[Group('table_header_scope_attribute')]
class EmptyTableHeaderToTableDataTest extends UnitTestCase {

  use TableHeaderFilterTestTrait;

  /**
   * The HTML element validator service stub.
   */
  protected HtmlElementValidatorInterface&Stub $htmlElementValidator;

  /**
   * The class providing the filter to transform empty <th> into <td>.
   */
  protected EmptyTableHeaderToTableData $filter;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a test double for the HTML element validator service.
    $this->htmlElementValidator = $this->createMock(HtmlElementValidatorInterface::class);

    // Create the EmptyTableHeaderToTableData instance that needs testing.
    $this->filter = new EmptyTableHeaderToTableData([], 'table_header_scope_attribute_empty_table_header_to_table_data', ['provider' => 'test'], $this->htmlElementValidator);
    $this->filter->setStringTranslation($this->getStringTranslationStub());
  }

  /**
   * Tests that the filter correctly transforms empty <th> into <td>.
   */
  #[DataProvider('processProvider')]
  public function testProcess(string $text, string $expected, array $empty_th_contents = ['']): void {
    // Configure the stub to consider <th> elements empty based on test data.
    $this->configureEmptyElementValidator($empty_th_contents);

    $processed = $this->filter->process($text, 'en')->getProcessedText();

    $this->assertSame($expected, $processed);
  }

  /**
   * Provides data to test the filter processing.
   *
   * @return \Generator
   *   Yields test data arrays containing:
   *     - text: The input string to be processed.
   *     - expected: The expected output string.
   *     - empty_th_contents: Array of text contents that should be considered
   *       empty.
   */
  public static function processProvider(): \Generator {
    yield 'empty string' => [
      '',
      '',
    ];

    yield 'some text without a table' => [
      '<p>some text</p>',
      '<p>some text</p>',
    ];

    yield 'table with non-empty header' => [
      '<table><tbody><tr><th>header</th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><th>header</th><td>data</td></tr></tbody></table>',
    ];

    yield 'table with empty header' => [
      '<table><tbody><tr><th></th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><td></td><td>data</td></tr></tbody></table>',
    ];

    yield 'table with whitespace-only header' => [
      '<table><tbody><tr><th>   </th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><td>   </td><td>data</td></tr></tbody></table>',
      ['   '],
    ];

    yield 'table with th containing &nbsp;' => [
      '<table><tbody><tr><th>&nbsp;</th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><td>&nbsp;</td><td>data</td></tr></tbody></table>',
      ['&nbsp;'],
    ];

    yield 'table with th containing empty child elements' => [
      '<table><tbody><tr><th><span></span></th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><td><span></span></td><td>data</td></tr></tbody></table>',
    ];

    yield 'table with empty header element having attributes' => [
      '<table><tbody><tr><th id="header-id" class="header-class"></th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><td id="header-id" class="header-class"></td><td>data</td></tr></tbody></table>',
    ];

    yield 'table with empty header element having scope attribute' => [
      '<table><tbody><tr><th scope="col"></th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><td></td><td>data</td></tr></tbody></table>',
    ];

    yield 'table with multiple empty headers' => [
      '<table><tbody><tr><th></th><th></th><td>data1</td><td>data2</td></tr></tbody></table>',
      '<table><tbody><tr><td></td><td></td><td>data1</td><td>data2</td></tr></tbody></table>',
    ];

    yield 'table with multiple rows and empty headers' => [
      '<table><tbody><tr><th></th><td>data1</td></tr><tr><th></th><td>data2</td></tr></tbody></table>',
      '<table><tbody><tr><td></td><td>data1</td></tr><tr><td></td><td>data2</td></tr></tbody></table>',
    ];

    yield 'table with upper-left header element empty' => [
      '<table><thead><tr><th></th><th>col1</th></tr></thead><tbody><tr><th>row1</th><td>data</td></tr></tbody></table>',
      '<table><thead><tr><td></td><th>col1</th></tr></thead><tbody><tr><th>row1</th><td>data</td></tr></tbody></table>',
    ];

    yield 'table with mixed empty and non-empty headers' => [
      '<table><tbody><tr><th></th><th>header</th><th>   </th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><td></td><th>header</th><td>   </td><td>data</td></tr></tbody></table>',
      ['', '   '],
    ];

    yield 'table with nested elements in non-empty header' => [
      '<table><tbody><tr><th><strong>Bold</strong></th><td>data</td></tr></tbody></table>',
      '<table><tbody><tr><th><strong>Bold</strong></th><td>data</td></tr></tbody></table>',
    ];

    yield 'multiple tables with different empty headers' => [
      '<table><tbody><tr><th></th></tr></tbody></table><table><tbody><tr><th>header</th></tr></tbody></table>',
      '<table><tbody><tr><td></td></tr></tbody></table><table><tbody><tr><th>header</th></tr></tbody></table>',
    ];
  }

  /**
   * Ensures that filter tips are provided.
   */
  public function testTips(): void {
    $tips = $this->filter->tips();

    $this->assertNotEmpty($tips);
    $this->assertIsString($tips);
  }

  /**
   * Tests that the validator service is properly called.
   */
  public function testValidatorServiceIntegration(): void {
    $html = '<table><tbody><tr><th>Header</th><th></th></tr></tbody></table>';

    // Ensure that the isElementContentEmpty method is called for <th> elements.
    $this->htmlElementValidator
      ->expects($this->atLeastOnce())
      ->method('isElementContentEmpty')
      ->with($this->callback(static function (\DOMElement $element): bool {
        return $element->tagName === 'th';
      }))
      ->willReturn(FALSE);

    $this->filter->process($html, 'en');
  }

}
