<?php

declare(strict_types=1);

namespace Drupal\Tests\table_header_scope_attribute\Unit;

use Drupal\table_header_scope_attribute\Plugin\Filter\TableHeaderScopeAttribute;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the TableHeaderScopeAttribute filter.
 */
#[CoversClass(TableHeaderScopeAttribute::class)]
#[Group('table_header_scope_attribute')]
class TableHeaderScopeAttributeTest extends UnitTestCase {

  /**
   * The class providing the filter to set scope attribute for table headers.
   */
  protected TableHeaderScopeAttribute $filter;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create the TableHeaderScopeAttribute instance that needs testing.
    $this->filter = new TableHeaderScopeAttribute([], 'table_header_scope_attribute', ['provider' => 'test']);
    $this->filter->setStringTranslation($this->getStringTranslationStub());
  }

  /**
   * Tests that the filter correctly sets the scope attribute to table headers.
   */
  #[DataProvider('processProvider')]
  public function testProcess(string $text, string $expected): void {
    $processed = $this->filter->process($text, 'en')->getProcessedText();

    $this->assertSame($expected, $processed);
  }

  /**
   * Provides data to test the filter processing.
   *
   * @return \Generator
   *   Yields test data arrays containing:
   *     - text: The input string to be processed.
   *     - expected: The expected output string.
   */
  public static function processProvider(): \Generator {
    yield 'empty string' => [
      '',
      '',
    ];

    yield 'some text without a table' => [
      '<p>some text</p>',
      '<p>some text</p>',
    ];

    yield 'table without header elements' => [
      '<table><tbody><tr><td>1</td><td>2</td><td>3</td></tr></tbody></table>',
      '<table><tbody><tr><td>1</td><td>2</td><td>3</td></tr></tbody></table>',
    ];

    yield 'table only one header element' => [
      '<table><thead><tr><th>1</th></tr></thead></table>',
      '<table><thead><tr><th>1</th></tr></thead></table>',
    ];

    yield 'table only multiple header elements' => [
      '<table><thead><tr><th>1</th><th>2</th><th>3</th></tr></thead></table>',
      '<table><thead><tr><th>1</th><th>2</th><th>3</th></tr></thead></table>',
    ];

    yield 'table one column' => [
      '<table><thead><tr><th>label</th></tr></thead><tbody><tr><td>value 1</td></tr><tr><td>value 2</td></tr></tbody></table>',
      '<table><thead><tr><th scope="col">label</th></tr></thead><tbody><tr><td>value 1</td></tr><tr><td>value 2</td></tr></tbody></table>',
    ];

    yield 'table multiple columns' => [
      '<table><thead><tr><th>label a</th><th>label b</th></tr></thead><tbody><tr><td>value 1</td><td>value 2</td></tr></tbody></table>',
      '<table><thead><tr><th scope="col">label a</th><th scope="col">label b</th></tr></thead><tbody><tr><td>value 1</td><td>value 2</td></tr></tbody></table>',
    ];

    yield 'table one row' => [
      '<table><tbody><tr><th>label</th><td>value 1</td><td>value 2</td></tr></tbody></table>',
      '<table><tbody><tr><th scope="row">label</th><td>value 1</td><td>value 2</td></tr></tbody></table>',
    ];

    yield 'table multiple rows' => [
      '<table><tbody><tr><th>label a</th><td>value 1</td><td>value 2</td></tr><tr><th>label b</th><td>value 3</td><td>value 4</td></tr></tbody></table>',
      '<table><tbody><tr><th scope="row">label a</th><td>value 1</td><td>value 2</td></tr><tr><th scope="row">label b</th><td>value 3</td><td>value 4</td></tr></tbody></table>',
    ];

    yield 'table multiple columns and rows' => [
      '<table><thead><tr><th>label a</th><th>label b</th></tr></thead><tbody><tr><th>label 1</th><td>value 1</td></tr><tr><th>label 2</th><td>value 2</td></tr></tbody></table>',
      '<table><thead><tr><th scope="col">label a</th><th scope="col">label b</th></tr></thead><tbody><tr><th scope="row">label 1</th><td>value 1</td></tr><tr><th scope="row">label 2</th><td>value 2</td></tr></tbody></table>',
    ];

    yield 'table with colspan' => [
      '<table><thead><tr><th colspan="2">label</th></tr></thead><tbody><tr><td>value 1</td><td>value 2</td></tr></tbody></table>',
      '<table><thead><tr><th colspan="2" scope="colgroup">label</th></tr></thead><tbody><tr><td>value 1</td><td>value 2</td></tr></tbody></table>',
    ];

    yield 'table with rowspan' => [
      '<table><tbody><tr><th rowspan="2">label</th><td>value 1</td></tr><tr><td>value 2</td></tr></tbody></table>',
      '<table><tbody><tr><th rowspan="2" scope="rowgroup">label</th><td>value 1</td></tr><tr><td>value 2</td></tr></tbody></table>',
    ];

    yield 'table with colspan and rowspan' => [
      '<table><thead><tr><th></th><th colspan="2">label a</th></tr></thead><tbody><tr><th rowspan="2">label b</th><td>value 1</td><td>value 2</td></tr><tr><td>value 3</td><td>value 4</td></tr></tbody></table>',
      '<table><thead><tr><th scope="col"></th><th colspan="2" scope="colgroup">label a</th></tr></thead><tbody><tr><th rowspan="2" scope="rowgroup">label b</th><td>value 1</td><td>value 2</td></tr><tr><td>value 3</td><td>value 4</td></tr></tbody></table>',
    ];
  }

  /**
   * Ensures that filter tips are provided.
   */
  public function testTips(): void {
    $tips = $this->filter->tips();

    $this->assertNotEmpty($tips);
    $this->assertIsString($tips);
  }

}
