<?php

namespace Drupal\Tests\table_of_contents\Kernel;

use Drupal\Core\Plugin\Context\EntityContext;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\table_of_contents\Traits\TableOfContentsTestTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;

/**
 * Tests the TextLongFieldTocBlock plugin.
 *
 * @group table_of_contents
 * @covers \Drupal\table_of_contents\Plugin\Block\TextLongFieldTocBlock
 */
class TextLongFieldTocBlockTest extends KernelTestBase {

  use TableOfContentsTestTrait;
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'field',
    'filter',
    'node',
    'system',
    'table_of_contents',
    'text',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('node', ['node_access']);
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installConfig(['filter', 'node', 'system']);

    // Create a user for node creation.
    $this->setUpCurrentUser();
  }

  /**
   * Helper to build the TOC block for a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param string $field_name
   *   The field name (default: 'field_toc_body').
   *
   * @return array|null
   *   The render array or null if no TOC.
   */
  protected function buildTocBlock($node, string $field_name = 'field_toc_body') {
    $plugin_id = 'text_long_field_toc_block:node:toc_test:' . $field_name;

    /** @var \Drupal\Core\Block\BlockManagerInterface $block_manager */
    $block_manager = \Drupal::service('plugin.manager.block');

    /** @var \Drupal\table_of_contents\Plugin\Block\TextLongFieldTocBlock $block */
    $block = $block_manager->createInstance($plugin_id);

    // Set the entity context.
    $context = EntityContext::fromEntity($node);
    $block->setContext('entity', $context);

    return $block->build();
  }

  /**
   * Tests basic heading extraction with h2 selector.
   */
  public function testBasicHeadingExtraction(): void {
    $this->createTocEnabledContentType('h2');

    $html = '<h2 id="intro">Introduction</h2><p>Some text</p><h2 id="details">Details</h2>';
    $node = $this->createTocTestNode($html);

    $build = $this->buildTocBlock($node);

    $this->assertNotNull($build);
    $this->assertEquals('item_list__table_of_contents', $build['#theme']);
    $this->assertCount(2, $build['#items']);

    // Verify first item.
    $this->assertEquals('Introduction', $build['#items'][0]['#title']);
    $this->assertEquals('intro', $build['#items'][0]['#url']->getOption('fragment'));

    // Verify second item.
    $this->assertEquals('Details', $build['#items'][1]['#title']);
    $this->assertEquals('details', $build['#items'][1]['#url']->getOption('fragment'));
  }

  /**
   * Tests custom CSS selector (h3 instead of h2).
   */
  public function testCustomCssSelector(): void {
    $this->createTocEnabledContentType('h3');

    $html = '<h2 id="ignored">Ignored H2</h2><h3 id="included">Included H3</h3>';
    $node = $this->createTocTestNode($html);

    $build = $this->buildTocBlock($node);

    $this->assertNotNull($build);
    $this->assertCount(1, $build['#items']);
    $this->assertEquals('Included H3', $build['#items'][0]['#title']);
  }

  /**
   * Tests ID generation for headings without IDs.
   */
  public function testIdGenerationForHeadingsWithoutIds(): void {
    $this->createTocEnabledContentType('h2');

    $html = '<h2>No ID Heading</h2>';
    $node = $this->createTocTestNode($html);

    $build = $this->buildTocBlock($node);

    $this->assertNotNull($build);
    $this->assertCount(1, $build['#items']);

    // Verify ID was generated with toc- prefix.
    $fragment = $build['#items'][0]['#url']->getOption('fragment');
    $this->assertStringStartsWith('toc-', $fragment);

    // Verify the link has invalid-id class.
    $this->assertContains('toc-link-invalid-id', $build['#items'][0]['#attributes']['class']);
  }

  /**
   * Tests that existing IDs are preserved.
   */
  public function testExistingIdsArePreserved(): void {
    $this->createTocEnabledContentType('h2');

    $html = '<h2 id="my-custom-id">Custom ID Heading</h2>';
    $node = $this->createTocTestNode($html);

    $build = $this->buildTocBlock($node);

    $fragment = $build['#items'][0]['#url']->getOption('fragment');
    $this->assertEquals('my-custom-id', $fragment);

    // Should NOT have invalid-id class.
    $this->assertNotContains('toc-link-invalid-id', $build['#items'][0]['#attributes']['class']);
  }

  /**
   * Tests multi-value field content is concatenated.
   */
  public function testMultiValueFieldConcatenation(): void {
    $this->createTocEnabledContentType('h2');

    // Modify field to allow multiple values.
    $field_storage = FieldStorageConfig::loadByName('node', 'field_toc_body');
    $field_storage->setCardinality(-1);
    $field_storage->save();

    $node = $this->createNode([
      'type' => 'toc_test',
      'title' => 'Multi Value Test',
      'field_toc_body' => [
        ['value' => '<h2 id="first">First Value</h2>'],
        ['value' => '<h2 id="second">Second Value</h2>'],
      ],
    ]);

    $build = $this->buildTocBlock($node);

    $this->assertCount(2, $build['#items']);
    $this->assertEquals('First Value', $build['#items'][0]['#title']);
    $this->assertEquals('Second Value', $build['#items'][1]['#title']);
  }

  /**
   * Tests empty field returns no TOC.
   */
  public function testEmptyFieldReturnsNoToc(): void {
    $this->createTocEnabledContentType('h2');

    $node = $this->createNode([
      'type' => 'toc_test',
      'title' => 'Empty Field Test',
      'field_toc_body' => [],
    ]);

    $build = $this->buildTocBlock($node);

    $this->assertNull($build);
  }

  /**
   * Tests field with no matching headings returns no TOC.
   */
  public function testNoMatchingHeadingsReturnsNoToc(): void {
    $this->createTocEnabledContentType('h2');

    $html = '<p>Just a paragraph, no headings.</p><h3>Wrong level heading</h3>';
    $node = $this->createTocTestNode($html);

    $build = $this->buildTocBlock($node);

    $this->assertNull($build);
  }

  /**
   * Tests that heading text content is correctly captured.
   */
  public function testHeadingTextContentIsCaptured(): void {
    $this->createTocEnabledContentType('h2');

    // Test with nested elements inside heading.
    $html = '<h2 id="nested">Heading with <strong>bold</strong> text</h2>';
    $node = $this->createTocTestNode($html);

    $build = $this->buildTocBlock($node);

    $this->assertNotNull($build);
    // Text content should include text from nested elements.
    $this->assertEquals('Heading with bold text', $build['#items'][0]['#title']);
  }

}
