<?php

namespace Drupal\Tests\table_of_contents\Traits;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;

/**
 * Provides helper methods for Table of Contents tests.
 */
trait TableOfContentsTestTrait {

  use ContentTypeCreationTrait;
  use NodeCreationTrait;

  /**
   * Creates a content type with a text_long field that has TOC enabled.
   *
   * @param string $css_selector
   *   The CSS selector for TOC items (default: 'h2').
   *
   * @return array
   *   Array with 'content_type', 'field_storage', and 'field_config' keys.
   */
  protected function createTocEnabledContentType(string $css_selector = 'h2'): array {
    // Create content type.
    $content_type = $this->createContentType(['type' => 'toc_test']);

    // Create field storage.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_toc_body',
      'entity_type' => 'node',
      'type' => 'text_long',
    ]);
    $field_storage->save();

    // Create field instance with TOC enabled.
    $field_config = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'toc_test',
      'label' => 'TOC Body',
    ]);
    // Enable TOC block for this field.
    $field_config->setThirdPartySetting('table_of_contents', 'toc_block', TRUE);
    $field_config->setThirdPartySetting('table_of_contents', 'toc_selector', $css_selector);
    $field_config->save();

    // Clear caches so the block deriver can find the new field.
    \Drupal::service('entity_field.manager')->clearCachedFieldDefinitions();
    \Drupal::service('plugin.manager.block')->clearCachedDefinitions();

    return [
      'content_type' => $content_type,
      'field_storage' => $field_storage,
      'field_config' => $field_config,
    ];
  }

  /**
   * Creates a node with the specified HTML content in the TOC body field.
   *
   * @param string $html_content
   *   The HTML content containing headings.
   * @param string|null $format
   *   The text format to use (default: NULL for no processing).
   *
   * @return \Drupal\node\NodeInterface
   *   The created node.
   */
  protected function createTocTestNode(string $html_content, ?string $format = NULL) {
    $field_value = ['value' => $html_content];
    if ($format !== NULL) {
      $field_value['format'] = $format;
    }
    return $this->createNode([
      'type' => 'toc_test',
      'title' => 'TOC Test Node',
      'field_toc_body' => $field_value,
    ]);
  }

}
