/**
 * @file
 * Lightweight client-side table sorting for Drupal 11+ (jQuery 4 compatible).
 */

(function ($, Drupal, once) {
  'use strict';

  Drupal.behaviors.tablesorterJs = {
    attach: function (context) {
      once('tablesorter-js', 'table.tablesorter', context).forEach(function (table) {
        var $table = $(table);
        var $headers = $table.find('thead th');
        var striping = $table.data('striping') === 1;

        // Add sortable styling to headers.
        $headers.css('cursor', 'pointer').attr('tabindex', '0');

        // Handle click and keyboard events on headers.
        $headers.on('click keypress', function (e) {
          // For keyboard, only respond to Enter or Space.
          if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) {
            return;
          }

          var $th = $(this);
          var index = $th.index();
          var asc = !$th.hasClass('tablesorter-headerAsc');

          // Reset all headers.
          $headers
            .removeClass('tablesorter-headerAsc tablesorter-headerDesc')
            .removeAttr('aria-sort');

          // Set current header state.
          $th.addClass(asc ? 'tablesorter-headerAsc' : 'tablesorter-headerDesc')
            .attr('aria-sort', asc ? 'ascending' : 'descending');

          // Sort rows.
          var $tbody = $table.find('tbody');
          var rows = $tbody.find('tr').get();

          rows.sort(function (a, b) {
            var $aCell = $(a).children('td, th').eq(index);
            var $bCell = $(b).children('td, th').eq(index);
            var aVal = $aCell.text().trim();
            var bVal = $bCell.text().trim();

            // Try numeric comparison (handles currency, commas).
            var aNum = parseFloat(aVal.replace(/[,$%]/g, ''));
            var bNum = parseFloat(bVal.replace(/[,$%]/g, ''));

            if (!isNaN(aNum) && !isNaN(bNum)) {
              return asc ? aNum - bNum : bNum - aNum;
            }

            // Fall back to string comparison.
            return asc ? aVal.localeCompare(bVal) : bVal.localeCompare(aVal);
          });

          $tbody.append(rows);

          // Re-apply zebra striping if enabled (matches core's data-striping).
          if (striping) {
            $tbody.find('tr').each(function (i) {
              $(this)
                .removeClass('odd even')
                .addClass(i % 2 === 0 ? 'odd' : 'even');
            });
          }
        });
      });
    }
  };

})(jQuery, Drupal, once);
