<?php

namespace Drupal\tailwind_merge_classes\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a configuration form for the Tailwind class prefix.
 */
class TailwindPrefixForm extends ConfigFormBase {

  /**
   * Render cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cacheRender;

  /**
   * Dynamic page cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cacheDynamicPage;

  /**
   * Page cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cachePage;

  /**
   * Constructs a TailwindPrefixForm object.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheRender
   *   Render cache service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheDynamicPage
   *   Dynamic page cache service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cachePage
   *   Page cache service.
   */
  public function __construct(
    CacheBackendInterface $cacheRender,
    CacheBackendInterface $cacheDynamicPage,
    CacheBackendInterface $cachePage,
  ) {
    $this->cacheRender = $cacheRender;
    $this->cacheDynamicPage = $cacheDynamicPage;
    $this->cachePage = $cachePage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('cache.render'),
      $container->get('cache.dynamic_page_cache'),
      $container->get('cache.page')
    );
  }

  /**
   * Returns the names of editable configuration objects.
   *
   * @return string[]
   *   Editable configuration names.
   */
  protected function getEditableConfigNames(): array {
    return ['tailwind_merge_classes.settings'];
  }

  /**
   * Returns the unique ID of the form.
   *
   * @return string
   *   Form ID.
   */
  public function getFormId(): string {
    return 'tailwind_prefix_form';
  }

  /**
   * Builds the configuration form.
   *
   * @param array<string, mixed> $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return array<string, mixed>
   *   Form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('tailwind_merge_classes.settings');

    // Textfield for the Tailwind class prefix.
    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tailwind class prefix'),
      '#description' => $this->t('Optional prefix to add before classes, for example ‘tw’ or ‘tw-’. Both will work the same way, like ‘tw-red-500’.'),
      '#default_value' => $config->get('prefix') ?? '',
      '#maxlength' => 32,
    ];

    // Standard submit button.
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Validates the prefix field input.
   *
   * @param array<string, mixed> $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return void
   *   Returns nothing.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $prefix = (string) $form_state->getValue('prefix', '');
    if ($prefix !== '' && !preg_match('/^[A-Za-z0-9_-]*$/', $prefix)) {
      $form_state->setErrorByName('prefix', $this->t('Prefix can only contain letters, numbers, hyphen and underscore.'));
    }
  }

  /**
   * Handles form submission.
   *
   * @param array<string, mixed> $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return void
   *   Returns nothing.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $prefix = (string) $form_state->getValue('prefix', '');

    // Save the user-provided prefix to the config.
    $this->configFactory()->getEditable('tailwind_merge_classes.settings')
      ->set('prefix', $prefix)
      ->save();

    // Clear caches using deleteAll().
    $this->cacheRender->deleteAll();
    $this->cacheDynamicPage->deleteAll();
    $this->cachePage->deleteAll();

    $this->messenger()->addStatus($this->t('Prefix saved and caches cleared.'));
  }

}
