<?php

namespace Drupal\tailwind_merge_classes\Service;

use TailwindMerge\TailwindMerge;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for merging Tailwind CSS classes intelligently.
 *
 * Uses the Tailwind Merge library and optionally applies a prefix configured
 * in Drupal settings.
 */
class TailwindMergeService {

  /**
   * Singleton instance of TailwindMerge for standard merging.
   *
   * @var \TailwindMerge\TailwindMerge
   */
  public $twMergeInstance;

  /**
   * Factory instance of TailwindMerge for custom configuration.
   *
   * @var \TailwindMerge\TailwindMerge
   */
  public $twMergeFactory;

  /**
   * Prefix applied to Tailwind classes if configured.
   *
   * @var string|null
   */
  protected $prefix;

  /**
   * Constructs the TailwindMergeService object.
   *
   * Loads the Tailwind Merge library instances and optional prefix from
   * configuration.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory to read the prefix setting.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    // Tailwind Merge library instances.
    $this->twMergeInstance = TailwindMerge::instance();
    $this->twMergeFactory = TailwindMerge::factory();

    // Load prefix from configuration (may be null or empty).
    $raw_prefix = $config_factory
      ->get('tailwind_merge_classes.settings')
      ->get('prefix') ?? NULL;

    // Normalize: add trailing hyphen if prefix is not empty and missing.
    if (!empty($raw_prefix) && substr($raw_prefix, -1) !== '-') {
      $raw_prefix .= '-';
    }

    $this->prefix = $raw_prefix;
  }

  /**
   * Merge Tailwind classes into a single string.
   *
   * If a prefix is set, merges only classes matching the prefix. Otherwise
   * merges all classes normally.
   *
   * @param string ...$classes
   *   Tailwind class strings to merge.
   *
   * @return string
   *   Merged Tailwind classes.
   */
  public function merge(...$classes): string {
    if (!empty($this->prefix)) {
      return $this->mergeWithPrefix(...$classes);
    }
    return $this->twMergeInstance->merge(...$classes);
  }

  /**
   * Merge Tailwind classes with the configured prefix into a string.
   *
   * Uses the factory instance to apply the prefix configuration.
   *
   * @param string ...$classes
   *   Tailwind class strings to merge.
   *
   * @return string
   *   Merged Tailwind classes.
   */
  public function mergeWithPrefix(...$classes): string {
    if (!empty($this->prefix)) {
      return $this->twMergeFactory
        ->withConfiguration(['prefix' => $this->prefix])
        ->make()
        ->merge(...$classes);
    }
    return $this->twMergeInstance->merge(...$classes);
  }

  /**
   * Merge Tailwind classes and return as an array.
   *
   * Splits the merged string into individual classes.
   *
   * @param string ...$classes
   *   Tailwind class strings to merge.
   *
   * @return array
   *   Array of merged Tailwind classes.
   */
  public function mergeAsArray(...$classes): array {
    if (!empty($this->prefix)) {
      return $this->mergeWithPrefixAsArray(...$classes);
    }
    return explode(' ', $this->twMergeInstance->merge(...$classes));
  }

  /**
   * Merge Tailwind classes with the configured prefix and return as array.
   *
   * Useful when individual class strings are needed separately.
   *
   * @param string ...$classes
   *   Tailwind class strings to merge.
   *
   * @return array
   *   Array of merged Tailwind classes.
   */
  public function mergeWithPrefixAsArray(...$classes): array {
    if (!empty($this->prefix)) {
      return explode(' ', $this->twMergeFactory
        ->withConfiguration(['prefix' => $this->prefix])
        ->make()
        ->merge(...$classes)
      );
    }
    return explode(' ', $this->twMergeInstance->merge(...$classes));
  }

}
