<?php

namespace Drupal\tailwind_merge_classes\Twig;

use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;
use Drupal\tailwind_merge_classes\Service\TailwindMergeService;

/**
 * Provides Twig functions for merging Tailwind classes in templates.
 *
 * Example usages:
 * 1. Normal classes (no prefix configured in module):
 *    {{ tw_merge('text-red-500', 'text-blue-500') }}
 *    → Returns 'text-blue-500'; classes are merged normally.
 *
 * 2. Prefixed classes (prefix set in module config):
 *    {{ tw_merge('tw-text-red-500', 'tw-text-blue-500', 'tw-p-4') }}
 *    → Returns 'tw-text-blue-500 tw-p-4'; only classes with the configured
 *       prefix are merged.
 *    * If you've added a prefix in your Tailwind CSS library config file and
 *      added the same prefix in this module's config form, this will work.
 */
class TailwindMergeClassesTwigExtension extends AbstractExtension {

  /**
   * Service instance used for merging Tailwind classes.
   *
   * @var \Drupal\tailwind_merge_classes\Service\TailwindMergeService
   */
  public $tailwindMergeService;

  /**
   * Constructs the Twig extension.
   *
   * @param \Drupal\tailwind_merge_classes\Service\TailwindMergeService $service
   *   Tailwind merge service injected via dependency injection.
   */
  public function __construct(TailwindMergeService $service) {
    $this->tailwindMergeService = $service;
  }

  /**
   * {@inheritdoc}
   *
   * Registers Twig functions provided by this extension.
   *
   * - `tw_merge`: merges Tailwind classes into a single string.
   * - `tw_merge_as_array`: merges Tailwind classes and
   *   returns them as an array.
   *
   * @return \Twig\TwigFunction[]
   *   Array of Twig functions.
   */
  public function getFunctions(): array {
    return [
      // Merge classes as string.
      new TwigFunction(
        'tw_merge',
        [$this, 'merge']
      ),
      // Merge classes as array.
      new TwigFunction(
        'tw_merge_as_array',
        [$this, 'mergeAsArray']
      ),
    ];
  }

  /**
   * Merge Tailwind classes into a string.
   *
   * Uses the configured prefix if set in module settings.
   *
   * @param string ...$classes
   *   Tailwind classes to merge.
   *
   * @return string
   *   Merged Tailwind classes as a single string.
   */
  public function merge(...$classes): string {
    return $this->tailwindMergeService->merge(...$classes);
  }

  /**
   * Merge Tailwind classes and return as array.
   *
   * Uses the configured prefix if set in module settings.
   *
   * @param string ...$classes
   *   Tailwind classes to merge.
   *
   * @return array
   *   Merged Tailwind classes as an array of strings.
   */
  public function mergeAsArray(...$classes): array {
    return $this->tailwindMergeService->mergeAsArray(...$classes);
  }

}
