<?php

namespace Drupal\Tests\tailwind_merge_classes\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\tailwind_merge_classes\Service\TailwindMergeService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;

/**
 * @coversDefaultClass \Drupal\tailwind_merge_classes\Service\TailwindMergeService
 * @group              tailwind_merge_classes
 *
 * Tests all main functionalities of the TailwindMergeService.
 * Normal classes for non-prefix methods, prefixed classes for prefix methods.
 */
class TailwindMergeServiceTest extends UnitTestCase {

  /**
   * TailwindMergeService instance with prefix.
   *
   * @var \Drupal\tailwind_merge_classes\Service\TailwindMergeService
   */
  protected $tailwindMergeServiceWithPrefix;

  /**
   * TailwindMergeService instance without prefix.
   *
   * @var \Drupal\tailwind_merge_classes\Service\TailwindMergeService
   */
  protected $tailwindMergeServiceWithoutPrefix;

  /**
   * Set up test environment.
   */
  protected function setUp(): void {
    parent::setUp();

    // Config mock with prefix 'tw-'.
    $config_with_prefix = $this->createMock(ImmutableConfig::class);
    $config_with_prefix->method('get')
      ->with('prefix')
      ->willReturn('tw-');

    $config_factory_with_prefix = $this->createMock(ConfigFactoryInterface::class);
    $config_factory_with_prefix->method('get')
      ->with('tailwind_merge_classes.settings')
      ->willReturn($config_with_prefix);

    $this->tailwindMergeServiceWithPrefix = new TailwindMergeService($config_factory_with_prefix);

    // Config mock with empty prefix.
    $config_without_prefix = $this->createMock(ImmutableConfig::class);
    $config_without_prefix->method('get')
      ->with('prefix')
      ->willReturn('');

    $config_factory_without_prefix = $this->createMock(ConfigFactoryInterface::class);
    $config_factory_without_prefix->method('get')
      ->with('tailwind_merge_classes.settings')
      ->willReturn($config_without_prefix);

    $this->tailwindMergeServiceWithoutPrefix = new TailwindMergeService($config_factory_without_prefix);
  }

  /**
   * Test merge() with normal classes (no prefix applied).
   *
   * @covers ::merge
   */
  public function testMergeNormalClasses(): void {
    $result = $this->tailwindMergeServiceWithoutPrefix->merge(
          'text-red-500',
          'text-blue-500'
      );
    $this->assertSame('text-blue-500', $result);
  }

  /**
   * Test mergeAsArray() with normal classes (no prefix applied).
   *
   * @covers ::mergeAsArray
   */
  public function testMergeAsArrayNormalClasses(): void {
    $result = $this->tailwindMergeServiceWithoutPrefix->mergeAsArray(
          'text-red-500',
          'text-blue-500'
      );
    $this->assertIsArray($result);
    $this->assertSame(['text-blue-500'], $result);
  }

  /**
   * Test merge() with prefixed classes (prefix applied from config).
   *
   * @covers ::merge
   */
  public function testMergePrefixedClasses(): void {
    $result = $this->tailwindMergeServiceWithPrefix->merge(
          'tw-text-red-500',
          'tw-text-blue-500'
      );
    $this->assertSame('tw-text-blue-500', $result);
  }

  /**
   * Test mergeAsArray() with prefixed classes (prefix applied from config).
   *
   * @covers ::mergeAsArray
   */
  public function testMergeAsArrayPrefixedClasses(): void {
    $result = $this->tailwindMergeServiceWithPrefix->mergeAsArray(
          'tw-text-red-500',
          'tw-text-blue-500'
      );
    $this->assertIsArray($result);
    $this->assertSame(['tw-text-blue-500'], $result);
  }

}
