<?php

namespace Drupal\tailwind_merge_classes\Service;

use TalesFromADev\TailwindMerge\TailwindMerge;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Module version 2.x with Tailwind CSS 4.x support.
 */

/**
 * Service for merging Tailwind CSS classes intelligently.
 *
 * Uses the Tailwind Merge library and optionally applies a prefix configured
 * in Drupal settings.
 */
class TailwindMergeService {

  /**
   * Singleton instance of TailwindMerge for standard merging.
   *
   * @var \TailwindMerge\TailwindMerge
   */
  public $twMergeInstance;

  /**
   * Singleton instance of TailwindMerge for standard merging.
   *
   * @var \TailwindMerge\TailwindMerge
   */
  public $twMergePrefixInstance;

  /**
   * Prefix applied to Tailwind classes if configured.
   *
   * @var string|null
   */
  protected $prefix;

  /**
   * Constructs the TailwindMergeService object.
   *
   * Loads the Tailwind Merge library instances and optional prefix from
   * configuration.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory to read the prefix setting.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    // Load prefix from configuration (may be null or empty).
    $raw_prefix = $config_factory
      ->get('tailwind_merge_classes.settings')
      ->get('prefix') ?? NULL;

    $this->prefix = $raw_prefix;

    // Tailwind Merge library instances.
    $this->twMergeInstance = new TailwindMerge();
    $this->twMergePrefixInstance = new TailwindMerge(['prefix' => $raw_prefix]);
  }

  /**
   * Merge Tailwind classes into a single string.
   *
   * If a prefix is set, merges only classes matching the prefix. Otherwise
   * merges all classes normally.
   *
   * @param string ...$classes
   *   Tailwind class strings to merge.
   *
   * @return string
   *   Merged Tailwind classes.
   */
  public function merge(...$classes): string {
    if (!empty($this->prefix)) {
      return $this->mergeWithPrefix(...$classes);
    }

    return $this->twMergeInstance->merge(...$classes);
  }

  /**
   * Merge Tailwind classes with the configured prefix into a string.
   *
   * Uses the factory instance to apply the prefix configuration.
   *
   * @param string ...$classes
   *   Tailwind class strings to merge.
   *
   * @return string
   *   Merged Tailwind classes.
   */
  public function mergeWithPrefix(...$classes): string {
    if (!empty($this->prefix)) {
      return $this->twMergePrefixInstance->merge(...$classes);
    }

    return $this->twMergeInstance->merge(...$classes);
  }

  /**
   * Merge Tailwind classes and return as an array.
   *
   * Splits the merged string into individual classes.
   *
   * @param string ...$classes
   *   Tailwind class strings to merge.
   *
   * @return array
   *   Array of merged Tailwind classes.
   */
  public function mergeAsArray(...$classes): array {
    if (!empty($this->prefix)) {
      return $this->mergeWithPrefixAsArray(...$classes);
    }

    return explode(' ', $this->twMergeInstance->merge(...$classes));
  }

  /**
   * Merge Tailwind classes with the configured prefix and return as array.
   *
   * Useful when individual class strings are needed separately.
   *
   * @param string ...$classes
   *   Tailwind class strings to merge.
   *
   * @return array
   *   Array of merged Tailwind classes.
   */
  public function mergeWithPrefixAsArray(...$classes): array {
    if (!empty($this->prefix)) {
      return explode(' ', $this->twMergePrefixInstance->merge(...$classes));
    }

    return explode(' ', $this->twMergeInstance->merge(...$classes));
  }

}
