<?php

namespace Drupal\Tests\tailwind_merge_classes\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\tailwind_merge_classes\Service\TailwindMergeService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;

/**
 * Module version 2.x with Tailwind CSS 4.x support.
 *
 * Tailwind CSS v4 support note:
 * In these test examples, Tailwind v4 utility classes are used.
 * Tailwind CSS v4 introduces new `-xs` variants such as `shadow-xs`,
 * `rounded-xs`, `blur-xs`, `backdrop-blur-xs`, and the new
 * `outline-hidden` utility.
 *
 * In Tailwind CSS v3, these utilities existed without the `-xs` suffix
 * (e.g. `shadow`, `rounded`, `blur`) and `outline-hidden` did not exist.
 *
 * @coversDefaultClass \Drupal\tailwind_merge_classes\Service\TailwindMergeService
 * @group tailwind_merge_classes
 *
 * Tests all main functionalities of the TailwindMergeService.
 * Normal classes for non-prefix methods, prefixed classes for prefix methods.
 */
class TailwindMergeServiceTest extends UnitTestCase {

  /**
   * TailwindMergeService instance with prefix.
   *
   * @var \Drupal\tailwind_merge_classes\Service\TailwindMergeService
   */
  protected $tailwindMergeServiceWithPrefix;

  /**
   * TailwindMergeService instance without prefix.
   *
   * @var \Drupal\tailwind_merge_classes\Service\TailwindMergeService
   */
  protected $tailwindMergeServiceWithoutPrefix;

  /**
   * Set up test environment.
   */
  protected function setUp(): void {
    parent::setUp();

    // Config mock with prefix 'tw'.
    $config_with_prefix = $this->createMock(ImmutableConfig::class);
    $config_with_prefix->method('get')
      ->with('prefix')
      ->willReturn('tw');

    $config_factory_with_prefix = $this->createMock(ConfigFactoryInterface::class);
    $config_factory_with_prefix->method('get')
      ->with('tailwind_merge_classes.settings')
      ->willReturn($config_with_prefix);

    $this->tailwindMergeServiceWithPrefix = new TailwindMergeService($config_factory_with_prefix);

    // Config mock with empty prefix.
    $config_without_prefix = $this->createMock(ImmutableConfig::class);
    $config_without_prefix->method('get')
      ->with('prefix')
      ->willReturn('');

    $config_factory_without_prefix = $this->createMock(ConfigFactoryInterface::class);
    $config_factory_without_prefix->method('get')
      ->with('tailwind_merge_classes.settings')
      ->willReturn($config_without_prefix);

    $this->tailwindMergeServiceWithoutPrefix = new TailwindMergeService($config_factory_without_prefix);
  }

  /**
   * Test merge() with normal classes (no prefix applied).
   *
   * @covers ::merge
   */
  public function testMergeNormalClasses(): void {
    $result = $this->tailwindMergeServiceWithoutPrefix->merge(
      'shadow-xs',
      'shadow-sm',
      'rounded-xs',
      'rounded-sm',
      'blur-xs',
      'blur-md',
      'backdrop-blur-xs',
      'backdrop-blur-lg',
      'outline-double',
      'outline-hidden',
    );
    $this->assertSame('shadow-sm rounded-sm blur-md backdrop-blur-lg outline-hidden', $result);
  }

  /**
   * Test mergeAsArray() with normal classes (no prefix applied).
   *
   * @covers ::mergeAsArray
   */
  public function testMergeAsArrayNormalClasses(): void {
    $result = $this->tailwindMergeServiceWithoutPrefix->mergeAsArray(
      'shadow-xs',
      'shadow-sm',
      'rounded-xs',
      'rounded-sm',
      'blur-xs',
      'blur-md',
    );
    $this->assertIsArray($result);
    $this->assertSame(['shadow-sm', 'rounded-sm', 'blur-md'], $result);
  }

  /**
   * Test merge() with prefixed classes (prefix applied from config).
   *
   * @covers ::merge
   */
  public function testMergePrefixedClasses(): void {
    $result = $this->tailwindMergeServiceWithPrefix->merge(
      'tw:backdrop-blur-xs',
      'tw:backdrop-blur-lg',
      'tw:outline-double',
      'tw:outline-hidden',
    );
    $this->assertSame('tw:backdrop-blur-lg tw:outline-hidden', $result);
  }

  /**
   * Test mergeAsArray() with prefixed classes (prefix applied from config).
   *
   * @covers ::mergeAsArray
   */
  public function testMergeAsArrayPrefixedClasses(): void {
    $result = $this->tailwindMergeServiceWithPrefix->mergeAsArray(
      'tw:shadow-xs',
      'tw:shadow-sm',
      'tw:rounded-xs',
      'tw:rounded-sm',
      'tw:blur-xs',
      'tw:blur-md',
      'tw:backdrop-blur-xs',
      'tw:backdrop-blur-lg',
    );
    $this->assertIsArray($result);
    $this->assertSame(['tw:shadow-sm', 'tw:rounded-sm', 'tw:blur-md', 'tw:backdrop-blur-lg'], $result);
  }

}
