<?php

namespace Drupal\tapis_app\TapisProvider;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Site\Settings;
use Drupal\node\NodeInterface;
use Drupal\tapis_app\DrupalIds;
use Drupal\tapis_app\Exception\TapisAppException;
use Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface;
use Drupal\tapis_system\DrupalIds as SystemDrupalIds;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides functionality to interact with the Tapis API for Tapis Apps.
 *
 * This class is responsible for creating, updating, and deleting Tapis Apps
 * through the Tapis API.
 *
 * @package Drupal\tapis_app\TapisProvider
 */
class TapisAppProvider implements TapisAppProviderInterface {

  const TAPIS_API_VERSION = "v3";

  /**
   * The HTTP client used for making requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The site settings.
   *
   * @var \Drupal\Core\Site\Settings
   */
  protected Settings $siteSettings;

  /**
   * The Tapis token provider.
   *
   * @var \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface
   */
  protected TapisTokenProviderInterface $tapisTokenProvider;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The Tapis site tenant provider.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */
  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * TapisAppProvider constructor.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Site\Settings $siteSettings
   *   The site settings.
   * @param \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface $tapisTokenProvider
   *   The Tapis token provider.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   */
  public function __construct(ClientInterface $httpClient,
                              Settings $siteSettings,
                              TapisTokenProviderInterface $tapisTokenProvider,
                              AccountProxyInterface $currentUser,
                              TapisSiteTenantProviderInterface $tapisSiteTenantProvider,
                              LoggerChannelFactoryInterface $loggerFactory) {
    $this->httpClient = $httpClient;
    $this->siteSettings = $siteSettings;
    $this->tapisTokenProvider = $tapisTokenProvider;
    $this->currentUser = $currentUser;
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
    $this->logger = $loggerFactory->get('tapis_app');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('settings'),
      $container->get('tapis_auth.tapis_token_provider'),
      $container->get("current_user"),
      $container->get("tapis_tenant.tapis_site_tenant_provider"),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function createAppVersion($tenantId, array $tapisDefinition, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      'json' => $tapisDefinition,
      "http_errors" => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("An error occurred when creating a new Tapis app version in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function getAppVersion($tenantId, $appId, $appVersion, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/$appId/$appVersion";

    $response = $this->httpClient->request('GET', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      "http_errors" => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("An error occurred when getting the Tapis app (id: '$appId', version: '$appVersion') in the '$tenant_tapis_id' tenant: $message");
    }

    return $response_body['result'];
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function updateAppVersion($tenantId, $appId, $appVersion, array $tapisDefinition, $uid = -1) {
    // Remove any non-update-able keys from $tapisDefinition
    // Note: the "FALSE" values are added so that array_diff_key would work.
    $non_updateable_keys = [
      "id" => FALSE,
      "owner" => FALSE,
      "enabled" => FALSE,
      'version' => FALSE,
    ];
    $tapisDefinition = array_diff_key($tapisDefinition, $non_updateable_keys);

    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/$appId/$appVersion";

    $response = $this->httpClient->request('PATCH', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        'json' => $tapisDefinition,
        "http_errors" => FALSE,
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("An error occurred when updating the Tapis app (id: '$appId', version: '$appVersion') in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function doesAppVersionExist($tenantId, $appId, $appVersion, $showDeleted = FALSE, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $showDeletedString = $showDeleted ? "true" : "false";

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/$appId/$appVersion/?showDeleted=$showDeletedString";

    $response = $this->httpClient->request('GET', $tapis_api_url,
      ['headers' => ['X-Tapis-Token' => $userJWT], "http_errors" => FALSE]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() === 200) {
      return TRUE;
    }
    elseif ($response->getStatusCode() === 404) {
      return FALSE;
    }
    else {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("An error occurred when checking if the Tapis app (id: '$appId', version: '$appVersion') exists in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function getAllVersionsForApp($tenantId, $appId, $uid = -1) {
    // @todo This doesn't implement the pagination-based request approach, since for any given app, we likely won't have 100s and 100s of different versions within Tapis.
    //   But need to confirm this before moving to production.
    // By not following a pagination approach,
    // we only need 1 request to get all versions for an app id.
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/?search=(id.eq.$appId)&limit=-1";

    $response = $this->httpClient->request('GET', $tapis_api_url, [
      'headers' => ['X-Tapis-Token' => $userJWT],
      "http_errors" => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("An error occurred when getting all the versions for the Tapis app (id: '$appId') in the '$tenant_tapis_id' tenant: $message");
    }

    $versions = [];
    foreach ($response_body['result'] as $appVersion) {
      $versions[] = $appVersion['version'];
    }
    return $versions;
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function convertAppToJSON($tenantId, NodeInterface $app) {
    if ($app->getEntityType()->id() !== 'node' || $app->bundle() !== DrupalIds::NODE_BUNDLE_APP) {
      // This should not happen.
      throw new TapisAppException("Failed to process the provided app");
    }

    $json = [];

    $json['id'] = $app->get(DrupalIds::APP_TAPIS_ID)->getValue()[0]['value'];
    $json['version'] = $app->get(DrupalIds::APP_VERSION)->getValue()[0]['value'];

    $owner_userid = $app->get(DrupalIds::APP_OWNER)->getValue()[0]['target_id'];
    $json['owner'] = $this->tapisTokenProvider->getTapisUsername($tenantId, $owner_userid);

    $appStatus = $app->get(DrupalIds::APP_STATUS)->getValue()[0]['value'];
    $json['enabled'] = $appStatus === "enabled";

    $defaultSystemEntity = $app->get(DrupalIds::APP_DEFAULT_SYSTEM)->first() ?? NULL;
    if ($defaultSystemEntity) {
      // $defaultSystemEntity =
      // $defaultSystemEntity->get("entity")->getTarget()->getValue();
      $defaultSystemEntityTarget = $defaultSystemEntity->get("entity");
      /** @var \Drupal\Core\Entity\Plugin\DataType\EntityReference $defaultSystemEntityTarget */
      $defaultSystemEntity = $defaultSystemEntityTarget->getTarget()->getValue();
    }

    $runtime = $app->get(DrupalIds::APP_RUNTIME)->getValue()[0]['value'];
    // $executable_path = $app->get(DrupalIds::APP_EXECUTABLE_PATH)
    // ->getValue()[0]['value'];
    if (isset($app->get(DrupalIds::APP_EXECUTABLE_PATH)->getValue()[0]['value'])) {
      $executable_path = $app->get(DrupalIds::APP_EXECUTABLE_PATH)->getValue()[0]['value'];
    }
    else {
      $executable_path = NULL;
    }
    if ($runtime === "EXECUTABLE") {
      if (!$defaultSystemEntity) {
        throw new TapisAppException("A default system is required for executable apps.");
      }
      // Check if the executable path is a non-empty string.
      if (!$executable_path) {
        throw new TapisAppException("An executable path is required for executable apps.");
      }
      // Get the runtimes available on the default system.
      $availableRuntimes = [];
      foreach ($defaultSystemEntity->get(SystemDrupalIds::SYSTEM_JOB_RUNTIMES)->getValue() ?? [] as $available_runtime) {
        $availableRuntimes[] = $available_runtime['value'];
      }

      if (count($availableRuntimes) === 0) {
        throw new TapisAppException("The default system does not have any available job runtimes.");
      }

      // If the default system supports singularity, we'll use that.
      // Otherwise, we'll use docker.
      if (in_array("SINGULARITY", $availableRuntimes)) {
        $json['runtime'] = "SINGULARITY";
        $json['runtimeOptions'] = ['SINGULARITY_RUN'];
        // Dummy container image name used for docker.
        $json['containerImage'] = "osp_placeholder.sif";
      }
      else {
        $json['runtime'] = "DOCKER";
        // Dummy container image name used for docker.
        $json['containerImage'] = "osp_placeholder";
      }
    }
    else {
      $json['runtime'] = $runtime;
      $runtimeOptions = [];
      foreach ($app->get(DrupalIds::APP_RUNTIME_OPTIONS)->getValue() as $runtimeOption) {
        $runtimeOptions[] = $runtimeOption['value'];
      }
      $json['runtimeOptions'] = $runtimeOptions ?: ['NONE'];

      $containerImageSource = $app->get(DrupalIds::APP_CONTAINER_IMAGE_SOURCE)->getValue()[0]['value'];
      if ($containerImageSource !== "image_uri") {
        // @todo For now, only direct image URIs are supported
        throw new TapisAppException("For now, only image URIs are supported as a source for the container images.");
      }
      $json['containerImage'] = $app->get(DrupalIds::APP_CONTAINER_IMAGE_URI)->getValue()[0]['value'];
    }

    $appType = $app->get(DrupalIds::APP_TYPE)->getValue()[0]['value'];

    $useBatchScheduler = $app->get(DrupalIds::APP_USE_BATCH_SCHEDULER)->getValue()[0]['value'];

    $jobType = "FORK";
    if ($useBatchScheduler) {
      $jobType = "BATCH";
    }

    $json['jobType'] = $jobType;

    $json['maxJobs'] = intval($app->get(DrupalIds::APP_MAX_JOBS)->getValue()[0]['value'] ?: -1);
    $json['maxJobsPerUser'] = intval($app->get(DrupalIds::APP_MAX_JOBS_PER_USER)->getValue()[0]['value'] ?: -1);

    // @todo Figure out the integration between Apps and Files
    $json['strictFileInputs'] = FALSE;

    $jobAttributes = [];

    // @todo commenting this out here, because we set this for each job at the job submission runtime, so we don't need to set it here.
    // only one of the below 2 lines should be uncommented
    // and used, if used here.
    // $jobAttributes['execSystemOutputDir'] =
    // '${JobWorkingDir}/jobs/${JobUUID}';
    // $jobAttributes['execSystemOutputDir'] =
    // '${JobWorkingDir}/jobs/${JobName}';.
    if ($defaultSystemEntity) {
      $jobAttributes["execSystemId"] = $defaultSystemEntity->get(SystemDrupalIds::SYSTEM_TAPIS_ID)->getValue()[0]['value'];
    }

    $appInputType = $app->get(DrupalIds::APP_INPUT_TYPE)->getValue()[0]['value'];

    $jobAttributes['parameterSet'] = [
      'appArgs' => [],
      'containerArgs' => [],
      'schedulerOptions' => [],
      'envVariables' => [],
    ];

    if ($appInputType === "none") {
      if ($runtime === "EXECUTABLE") {
        $jobAttributes['parameterSet']['appArgs'][] = [
          'name' => 'executable',
          'inputMode' => "FIXED",
          'arg' => $executable_path,
        ];
      }
    }
    elseif ($appInputType === "fixed_command") {
      $fixedCommand = $app->get(DrupalIds::APP_FIXED_COMMAND)->getValue()[0]['value'];
      if ($runtime === "EXECUTABLE") {
        $jobAttributes['parameterSet']['appArgs'][] = [
          'name' => 'executable',
          'inputMode' => "FIXED",
          'arg' => $executable_path,
        ];
      }
      $jobAttributes['parameterSet']['appArgs'][] = [
        'name' => 'fixed_command',
        'inputMode' => "FIXED",
        'arg' => $fixedCommand,
      ];
    }
    else {
      if ($runtime === "EXECUTABLE") {
        $jobAttributes['parameterSet']['appArgs'][] = [
          'name' => 'executable',
          'inputMode' => "FIXED",
          'arg' => $executable_path,
        ];
      }
      $jobAttributes['parameterSet']['appArgs'][] = [
        'name' => 'fixed_command',
        'inputMode' => "INCLUDE_ON_DEMAND",
      ];
      // Throw new TapisAppException("For app input type, only 'none'
      // and 'fixed command' are currently supported.");.
    }

    if ($appType === "web" || $appType === "vnc") {
      $app_container_port = $app->get(DrupalIds::APP_CONTAINER_PORT)->getValue()[0]['value'];
      if ($app_container_port && $runtime === "DOCKER") {
        $jobAttributes['parameterSet']['envVariables'][] = [
          'key' => 'OSP_APP_PORT',
          'value' => "$app_container_port",
        ];
      }
      elseif ($runtime === "SINGULARITY") {
        // The launcher will take care of setting the port for singularity apps
        // by replacing all occurrences of [OSP_APP_PORT]
        // with the actual port number,
        // at job launch time, on the compute system.
        $jobAttributes['parameterSet']['envVariables'][] = [
          'key' => 'OSP_APP_PORT',
          'value' => "[OSP_APP_PORT]",
        ];
      }
    }

    // Add any MPI settings (only applies
    // when appType = batch and runtime != EXECUTABLE)
    // the reason for the restrictions is because,
    // when its set, mpiCmd conflicts with cmdPrefix (used for our launcher)
    // and we're currently using cmdPrefix (ie., our launcher)
    // for web/vnc apps and executable apps.
    if ($appType === "batch" && $runtime !== "EXECUTABLE") {
      $usesMPI = boolval($app->get(DrupalIds::IS_MPI)->getValue()[0]['value']);
      $jobAttributes['isMpi'] = $usesMPI;
      // $mpiCommand = $app->get(DrupalIds::MPI_CMD)->getValue()[0]['value'];
      if (isset($app->get(DrupalIds::MPI_CMD)->getValue()[0]['value'])) {
        $mpiCommand = $app->get(DrupalIds::MPI_CMD)->getValue()[0]['value'];
      }
      else {
        $mpiCommand = NULL;
      }

      if ($usesMPI && $mpiCommand !== NULL) {
        $jobAttributes['mpiCmd'] = $mpiCommand;
      }
    }

    // Populate parameterSet for Tapis app from the App entity.
    if (!$app->get(DrupalIds::APP_ENV_VARS)->isEmpty()) {
      $envVarsString = $app->get(DrupalIds::APP_ENV_VARS)->getValue()[0]['value'];

      if ($envVarsString) {
        foreach (explode("\n", $envVarsString) as $envVarLine) {
          $envVarLine = trim($envVarLine);
          $envVarArr = explode("=", $envVarLine, 2);
          $envVar = ['key' => $envVarArr[0]];
          if (count($envVarArr) === 2) {
            $envVar['value'] = $envVarArr[1];
          }
          $jobAttributes['parameterSet']['envVariables'][] = $envVar;
        }
      }
    }

    if (!$app->get(DrupalIds::APP_CONTAINER_ARGS)->isEmpty()) {
      $containerArgsString = $app->get(DrupalIds::APP_CONTAINER_ARGS)->getValue()[0]['value'];
      if ($containerArgsString) {
        $containerArgsString = trim($containerArgsString);

        /*
         * containerArgsString will be a string of the form:
         * key1 value1
         * key2 value2
         * ...
         * where key is the name of the container argument and
         * value is the value of the container argument.
         * Each line will be a separate container argument,
         * and the value can be empty.
         * When a value is present, it will be separated
         * from the key by the first space character (' ').
         * Otherwise, the line will just be the key (and
         * if a space character is present at the end, it will be ignored).
         *
         * Note: the value may contain a space character, but the key cannot.
         */
        // Parse the containerArgsString into an array of container arguments.
        $containerArgLineIndex = 0;
        foreach (explode("\n", $containerArgsString) as $containerArgLine) {
          $containerArgLine = trim($containerArgLine);
          $jobAttributes['parameterSet']['containerArgs'][] = [
            'name' => "container_arg_$containerArgLineIndex",
            'inputMode' => "REQUIRED",
            "arg" => $containerArgLine,
          ];
          $containerArgLineIndex++;
        }
      }
    }

    $appSchedulerProfile = $app->get(DrupalIds::APP_SCHEDULER_PROFILE)->first() ?? NULL;
    if ($appSchedulerProfile) {
      // $appSchedulerProfile =
      // $appSchedulerProfile->get("entity")->getTarget()->getValue();
      $appSchedulerProfileTarget = $appSchedulerProfile->get("entity");
      /** @var \Drupal\Core\Entity\Plugin\DataType\EntityReference $appSchedulerProfileTarget */
      $appSchedulerProfile = $appSchedulerProfileTarget->getTarget()->getValue();

      $appSchedulerProfileTapisId = $appSchedulerProfile->get(SystemDrupalIds::SYSTEM_SCHEDULER_PROFILE_TAPIS_ID)->getValue()[0]['value'];
      $jobAttributes['parameterSet']['schedulerOptions'][] = [
        'name' => '--tapis-profile',
        'inputMode' => "FIXED",
        "arg" => "--tapis-profile $appSchedulerProfileTapisId",
      ];
    }

    if (!$app->get(DrupalIds::APP_BATCH_SCHED_OPTIONS)->isEmpty()) {
      $schedulerArgsString = $app->get(DrupalIds::APP_BATCH_SCHED_OPTIONS)->getValue()[0]['value'];
      if ($schedulerArgsString) {
        $schedulerArgsString = trim($schedulerArgsString);
        /*
         * $schedulerArgsString will be a string of the form:
         * key1 value1
         * key2 value2
         * ...
         * where key is the name of the scheduler arg and
         *  value is the value of the scheduler argument.
         * Each line will be a separate scheduler argument, and
         *  the value can be empty.
         * When a value is present, it will be separated from
         *  the key by the first space character (' ').
         * Otherwise, the line will just be the key (and
         *  if a space character is present at the end, it will be ignored).
         *
         * Note: the value may contain a space character, but the key cannot.
         */
        // Parse the $schedulerArgsString into an array of scheduler arguments.
        $schedulerArgLineIndex = 0;
        foreach (explode("\n", $schedulerArgsString) as $schedulerArgLine) {
          $schedulerArgLine = trim($schedulerArgLine);
          $jobAttributes['parameterSet']['schedulerOptions'][] = [
            'name' => "scheduler_option_$schedulerArgLineIndex",
            'inputMode' => "FIXED",
            'arg' => $schedulerArgLine,
          ];
          $schedulerArgLineIndex++;
        }
      }
    }

    $jobAttributes['nodeCount'] = intval($app->get(DrupalIds::APP_NUM_NODES)->getValue()[0]['value'] ?: 1);
    $jobAttributes['coresPerNode'] = intval($app->get(DrupalIds::APP_CORES_PER_NODE)->getValue()[0]['value'] ?: 1);
    $jobAttributes['memoryMB'] = intval($app->get(DrupalIds::APP_MEMORY_MB)->getValue()[0]['value'] ?: 100);
    $jobAttributes['maxMinutes'] = intval($app->get(DrupalIds::APP_MAX_MINUTES)->getValue()[0]['value'] ?: 10);

    $json['jobAttributes'] = $jobAttributes;
    return $json;
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function shareAppWithTenant($tenantId, $appId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/share_public/$appId/";

    $response = $this->httpClient->request(
      'POST',
      $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        'http_errors' => FALSE,
      ]
    );
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("An error occurred when sharing the Tapis app (id: '$appId') with the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function unshareAppWithTenant($tenantId, $appId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/unshare_public/$appId/";

    $response = $this->httpClient->request(
      'POST',
      $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        'http_errors' => FALSE,
      ]
    );
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("An error occurred when un-sharing the Tapis app (id: '$appId') with the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function getShareAppWithTenant($tenantId, $appId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/share/$appId";

    $response = $this->httpClient->request('GET', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("Could not get the share list for the Tapis '$appId' system with the '$tenant_tapis_id' tenant: $message");
    }
    else {
      return $response_body['result'];
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function shareAppWithUser($tenantId, $appId, $shareUserId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);
    $username = $this->tapisTokenProvider->getTapisUsername($tenantId, $shareUserId);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/share/$appId/";

    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
        'json' => [
          "users" => [$username],
        ],
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug("StatusCode = " . $response->getStatusCode());
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("Could not grant the Tapis '$appId' app permission to the user '$username' in the '$tenant_tapis_id' tenant: $message");
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_app\Exception\TapisAppException
   */
  public function unshareAppWithUser($tenantId, $appId, $shareUserId, $uid = -1) {
    if ($uid === -1) {
      $uid = $this->currentUser->id();
    }
    $userJWT = $this->tapisTokenProvider->getUserToken($tenantId, $uid);
    $username = $this->tapisTokenProvider->getTapisUsername($tenantId, $shareUserId);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $tenant_tapis_id = $tenantInfo['tapis_id'];

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/apps/unshare/$appId/";

    $response = $this->httpClient->request('POST', $tapis_api_url,
      [
        'headers' => ['X-Tapis-Token' => $userJWT],
        "http_errors" => FALSE,
        'json' => [
          "users" => [$username],
        ],
      ]);
    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    // \Drupal::logger('tapis_system')
    // ->debug(print_r($response_body, TRUE));.
    $this->logger->debug("StatusCode = " . $response->getStatusCode());
    $this->logger->debug(print_r($response_body, TRUE));

    if ($response->getStatusCode() !== 200 && $response->getStatusCode() !== 201) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAppException("Could not unshare the Tapis '$appId' system permission to the user '$username' in the '$tenant_tapis_id' tenant: $message");
    }
  }

}
