<?php

namespace Drupal\tapis_app\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\Markup;
use Drupal\node\NodeInterface;
use Drupal\tapis_app\DrupalIds;
use Drupal\tapis_app\TapisProvider\TapisAppProviderInterface;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides a form for sharing/unsharing a Tapis App with a Tapis tenant.
 */
class TapisAppTenantSharingForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The messenger interface.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The Tapis app provider.
   *
   * @var \Drupal\tapis_app\TapisProvider\TapisAppProviderInterface
   */
  protected TapisAppProviderInterface $tapisAppProvider;

  /**
   * The Tapis site tenant provider.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */
  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * TapisAppTenantSharingForm constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\tapis_app\TapisProvider\TapisAppProviderInterface $tapisAppProvider
   *   The Tapis app provider.
   * @param \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface $tapisSiteTenantProvider
   *   The Tapis site tenant provider.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, MessengerInterface $messenger, TapisAppProviderInterface $tapisAppProvider, TapisSiteTenantProviderInterface $tapisSiteTenantProvider) {
    $this->entityTypeManager = $entityTypeManager;
    $this->messenger = $messenger;
    $this->tapisAppProvider = $tapisAppProvider;
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('messenger'),
      $container->get('tapis_app.tapis_app_provider'),
      $container->get("tapis_tenant.tapis_site_tenant_provider")
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'tapis_app_tenantsharing_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL) {
    $form['#tree'] = TRUE;

    if (!$node) {
      return;
    }

    if ($node->bundle() !== DrupalIds::NODE_BUNDLE_APP) {
      throw new NotFoundHttpException();
    }

    // Check if the app's Tapis tenant's Tapis site is in maintenance mode.
    $tenantId = $node->get(DrupalIds::APP_TENANT)->first()->getValue()['target_id'];
    if ($this->tapisSiteTenantProvider->isTenantInMaintenanceMode($tenantId)) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => '<p>This app cannot be shared because its site is in maintenance mode.</p>',
      ];
      return $form;
    }

    $form_state->set("app_id", $node->id());

    $app_label = $node->label();

    $form['message_share'] = [
      '#type' => 'markup',
      '#markup' => "<p>Sharing this app will enable site users other than <b>$app_label</b>'s owner to also use the app (<b><i>only</i></b> if they satisfy the app's access control policies).</p>",
    ];

    $form['message_unshare'] = [
      '#type' => 'markup',
      '#markup' => "<p>Unsharing this app will prevent site users other than <b>$app_label</b>'s owner from using the app (even if they satisfy the app's access control policies).</p>",
    ];

    $form['share_button'] = [
      '#type' => 'submit',
      '#value' => t('Share'),
    ];

    $form['unshare_button'] = [
      '#type' => 'submit',
      '#value' => t('Unshare'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $app_id = $form_state->get("app_id");

    /** @var \Drupal\node\NodeInterface $app */
    $app = $this->entityTypeManager->getStorage('node')->load($app_id);
    $appOwnerId = $app->getOwnerId();
    $tenantId = $app->get(DrupalIds::APP_TENANT)->first()->getValue()['target_id'];

    // Get the button that was clicked.
    $clicked_button = $form_state->getTriggeringElement()['#value']->render();
    if ($clicked_button === "Share") {
      $this->tapisAppProvider->shareAppWithTenant($tenantId, $app->get(DrupalIds::APP_TAPIS_ID)->getValue()[0]['value'], $appOwnerId);
      $this->messenger->addMessage(Markup::create("<b>" . $app->label() . "</b> has been shared with the tenant."));
    }
    else {
      $this->tapisAppProvider->unshareAppWithTenant($tenantId, $app->get(DrupalIds::APP_TAPIS_ID)->getValue()[0]['value'], $appOwnerId);
      $this->messenger->addMessage(Markup::create("<b>" . $app->label() . "</b> has been unshared with the tenant."));
    }
  }

}
