<?php

namespace Drupal\tapis_app\TapisProvider;

use Drupal\node\NodeInterface;

/**
 * Provides functionality to interact with the Tapis API for Tapis Apps.
 *
 * This interface is responsible for creating, updating, and deleting Tapis Apps
 * through the Tapis API.
 *
 * @package Drupal\tapis_app\TapisAppProviderInterface
 */
interface TapisAppProviderInterface {

  /**
   * Convert a Tapis app node to a JSON representation.
   *
   * It can be sent to the Tapis API.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param \Drupal\node\NodeInterface $app
   *   The Tapis app node.
   */
  public function convertAppToJSON($tenantId, NodeInterface $app);

  /**
   * Get all the versions for a Tapis app.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function getAllVersionsForApp($tenantId, $appId, $uid = -1);

  /**
   * Check if an App version exists.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param string $appVersion
   *   The version of the Tapis app.
   * @param bool $showDeleted
   *   (Optional) Whether to include deleted versions. Defaults to FALSE.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function doesAppVersionExist($tenantId, $appId, $appVersion, $showDeleted = FALSE, $uid = -1);

  /**
   * Get an App version's data from Tapis.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param string $appVersion
   *   The version of the Tapis app.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function getAppVersion($tenantId, $appId, $appVersion, $uid = -1);

  /**
   * Create an App version.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param array $tapisDefinition
   *   The Tapis app definition.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function createAppVersion($tenantId, array $tapisDefinition, $uid = -1);

  /**
   * Update an App version in Tapis.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param string $appVersion
   *   The version of the Tapis app.
   * @param array $tapisDefinition
   *   The updated Tapis app definition.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function updateAppVersion($tenantId, $appId, $appVersion, array $tapisDefinition, $uid = -1);

  /**
   * Share an App with a Tapis tenant.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function shareAppWithTenant($tenantId, $appId, $uid = -1);

  /**
   * Unshare an App with a Tapis tenant.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function unshareAppWithTenant($tenantId, $appId, $uid = -1);

  /**
   * Share a system with a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function getShareAppWithTenant($tenantId, $appId, $uid = -1);

  /**
   * Share an App with a user.
   *
   * @param string $tenantId
   *   The ID of the Tapis tenant.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param int $shareUserId
   *   The share user ID.
   * @param int $uid
   *   (Optional) The user ID. Defaults to -1.
   */
  public function shareAppWithUser($tenantId, $appId, $shareUserId, $uid = -1);

  /**
   * Share a system with a Tapis tenant.
   *
   * @param string $tenantId
   *   The Tapis tenant ID.
   * @param string $appId
   *   The ID of the Tapis app.
   * @param int $shareUserId
   *   The share user ID.
   * @param int $uid
   *   The user ID (optional, defaults to -1).
   */
  public function unshareAppWithUser($tenantId, $appId, $shareUserId, $uid = -1);

}
