<?php

namespace Drupal\tapis_auth\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\tapis_auth\Constants;
use Drupal\tapis_auth\TapisTokenInterface;
use Drupal\user\EntityOwnerTrait;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Defines the token entity class.
 *
 * @ContentEntityType(
 *   id = "tapis_token",
 *   label = @Translation("Token"),
 *   label_collection = @Translation("Tokens"),
 *   label_singular = @Translation("token"),
 *   label_plural = @Translation("tokens"),
 *   label_count = @PluralTranslation(
 *     singular = "@count tokens",
 *     plural = "@count tokens",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\tapis_auth\TapisTokenListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *     "form" = {
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     }
 *   },
 *   base_table = "tapis_token",
 *   admin_permission = "administer tapis token",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "id",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   links = {
 *     "collection" = "/admin/content/tapis-token",
 *     "delete-form" = "/tapis/token/{tapis_token}/delete",
 *   },
 * )
 */
class TapisToken extends ContentEntityBase implements TapisTokenInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  // There are 2 types of tokens:
  // One type is for services, and the other type is for users.
  public const TYPE_SERVICE = 0;
  public const TYPE_USER = 1;

  /**
   * {@inheritdoc}
   */
  public function label() {
    if ($this->getSubjectType() === "user") {
      return $this->getOwner()->label() . " (user)";
    }
    else {
      return $this->getService() . " (service)";
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {

    $fields = parent::baseFieldDefinitions($entity_type);

    // The user id this token is for.
    // Only applies for *user* JWT tokens,
    // and only authorized users are allowed.
    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('User'))
      ->setRequired(FALSE)
      ->setSetting('target_type', 'user');

    // The tapis tenant this token exists in.
    $fields['tenant'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Tenant'))
      ->setRequired(TRUE)
      ->setSetting('target_type', 'node')
      ->setSetting('handler', 'default')
      ->setSetting('handler_settings', [
        'target_bundles' => [
          'tapis_tenant' => 'tapis_tenant',
        ],
      ]);

    // The type of token: 'service' (0) or 'user' (1)
    // 'User' tokens are for Drupal site users.
    // 'Service' tokens are for this module's Tapis authenticator service,
    // which itself needs a JWT token
    // in order to make API calls to the Tapis tokens service.
    $fields['type'] = BaseFieldDefinition::create("integer")
      ->setLabel(t("Type"))
      ->setRequired(TRUE);

    // The service name for this token.
    // Only applies for *service* JWT tokens (e.g., 'authenticator', etc.).
    // For now, the only value this could have at runtime is: 'authenticator'.
    $fields['service'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Service'))
      ->setRequired(FALSE)
      ->setDescription(t('The service username for this token (only applicable for service tokens).'));

    // The access token for this JWT.
    $fields['access_token'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Access token'))
      ->setRequired(TRUE)
      ->setDescription(t('The JWT access token for this user/service.'));

    // The refresh token for this JWT.
    // Note: currently this is not used.
    // When an access token is expired,
    // this module will just directly mint a new access token from Tapis,
    // instead of using the refresh token.
    // Tapis APIs support this token management design.
    $fields['refresh_token'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Refresh token'))
      ->setRequired(TRUE)
      ->setDescription(t('The JWT refresh token for this user/service.'));
    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);

    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      // Drupal always needs a value for the "uid" column,
      // but this doesn't affect any Tapis access control logic.
      $this->setOwnerId(0);
    }

    if ($this->getSubjectType() === "user") {
      // Ensure that the user id can authenticate w/ Tapis.
      /** @var \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface $tapisTokenProvider */
      $tapisTokenProvider = \Drupal::service(Constants::SERVICE_TAPIS_TOKEN_PROVIDER);
      if (!$tapisTokenProvider->canUserAccessTapisServices($this->getTenantId(), $this->getOwnerId())) {
        // You cannot save a JWT for a user that cannot access Tapis services.
        throw new AccessDeniedHttpException();
      }
    }
  }

  /**
   * Get the access token.
   */
  public function getAccessToken() {
    return $this->get("access_token")->getString();
  }

  /**
   * Get the refresh token.
   */
  public function getRefreshToken() {
    return $this->get("refresh_token")->getString();
  }

  /**
   * Set the access token.
   */
  public function setAccessToken($accessToken) {
    $this->set("access_token", $accessToken);
  }

  /**
   * Set the refresh token.
   */
  public function setRefreshToken($refreshToken) {
    $this->set("refresh_token", $refreshToken);
  }

  /**
   * Get the subject type for this token (e.g., 'user' or 'service')
   */
  public function getSubjectType() {
    $types = ['service', 'user'];
    return $types[$this->get("type")->getValue()[0]['value']];
  }

  /**
   * Get the service name for this token.
   */
  public function getService() {
    return $this->get("service")->getString();
  }

  /**
   * Get the tenant id for this token.
   */
  public function getTenantId() {
    return $this->get("tenant")->first()->getValue()['target_id'];
  }

  /**
   * Get the tenant info for this token.
   */
  public function getTenant() {
    $tenantId = $this->getTenantId();
    $tapisSiteTenantProvider = \Drupal::service("tapis_tenant.tapis_site_tenant_provider");
    return $tapisSiteTenantProvider->getTenantInfo($tenantId);
  }

}
