<?php

namespace Drupal\tapis_auth\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\NodeInterface;
use Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class GetTapisTenantStatusForm.
 *
 * Form for checking the status of a Tapis tenant.
 */
class GetTapisTenantStatusForm extends FormBase {
  /**
   * Node for Tenant.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected NodeInterface $tenant;

  /**
   * The Tapis token provider.
   *
   * @var \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface
   */
  protected TapisTokenProviderInterface $tapisTokenProvider;

  /**
   * GetTapisTenantStatusForm constructor.
   *
   * @param \Drupal\tapis_auth\TapisProvider\TapisTokenProviderInterface $tapisTokenProvider
   *   The Tapis token provider.
   */
  public function __construct(TapisTokenProviderInterface $tapisTokenProvider) {
    $this->tapisTokenProvider = $tapisTokenProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tapis_auth.tapis_token_provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'get_tapis_tenant_status_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeInterface $node = NULL): array {
    $this->tenant = $node;
    $form['test_button'] = [
      '#type' => 'button',
      '#button_type' => 'primary',
      '#value' => t('Check status'),
      '#prefix' => t('</br>'),
      '#suffix' => t('</br>'),
      '#ajax' => [
        'callback' => '::ajax_handler',
        'event' => 'click',
      ],
    ];

    // Attach the library for pop-up dialogs/modals.
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
    return $form;
  }

  /**
   * Ajax handler for the "Check status" button.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function ajax_handler(array $form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    // Try to create a new authenticator token,
    // and if any exception occurs, post a message.
    try {
      $tokens = $this->tapisTokenProvider->getNewAuthenticatorServiceToken($this->tenant->id());
      // Verify that $tokens has both keys: access_token and refresh_token.
      if (array_key_exists('access_token', $tokens) && array_key_exists('refresh_token', $tokens)) {
        $title = 'Success';
        $content['#markup'] = $this->t('This tenant is active.');
      }
      else {
        $title = 'Error';
        $content['#markup'] = $this->t("This tenant is NOT active. Please verify the tenant's information and try again.");
      }
    }
    catch (\Exception $e) {
      $title = 'Error';
      $content['#markup'] = $this->t("This tenant is NOT active. Please verify the tenant's information and try again.");
    }
    $response->addCommand(new OpenModalDialogCommand($title, $content,
      ['width' => '400', 'height' => '200']));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
  }

}
