<?php

namespace Drupal\tapis_auth\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provide the authentication settings for Tapis auth module.
 */
class SettingsForm extends ConfigFormBase {
  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'tapis_auth.config';

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * SettingsForm constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager,
                              ConfigFactoryInterface $config_factory) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'tapis_auth_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(static::SETTINGS);
    $integration_tests_secret_key_id = $config->get("integration_tests_secret_key_id");

    // Add a form field called 'integration_tests_secret' to the form
    // This should be a select dropdown,
    // and will store the target id to the key entity reference.
    $options = [];
    $all_keys = $this->entityTypeManager->getStorage('key')->loadMultiple();

    foreach ($all_keys as $key) {
      $options[$key->id()] = $key->label();
    }

    $form['enable_integration_tests'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Integration Tests'),
      '#description' => $this->t('Enable integration tests.'),
      '#default_value' => $config->get('enable_integration_tests') ?? FALSE,
    ];

    $form['integration_tests_secret_key_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Integration Tests Secret'),
      '#description' => $this->t('The secret to use for integration tests.'),
      '#options' => $options,
      '#default_value' => $integration_tests_secret_key_id ?? NULL,
      '#required' => FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory->getEditable(static::SETTINGS);
    $config->set("integration_tests_secret_key_id", $form_state->getValue("integration_tests_secret_key_id"));
    $config->set("enable_integration_tests", $form_state->getValue("enable_integration_tests"));
    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      static::SETTINGS,
    ];
  }

}
