<?php

namespace Drupal\tapis_auth\TapisProvider;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Site\Settings;
use Drupal\tapis_auth\Constants;
use Drupal\tapis_auth\Entity\TapisToken;
use Drupal\tapis_auth\Exception\TapisAuthException;
use Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface;
use Drupal\user\Entity\User;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Implements the TapisToken Provider interface.
 *
 * This class is used by the tapis_auth module for making API calls to Tapis.
 *
 * @package Drupal\tapis_app\TapisProvider
 */
class TapisTokenProvider implements TapisTokenProviderInterface {

  const TAPIS_API_VERSION = 'v3';

  const TOKEN_EXPIRY_WINDOW_SECS = 60 * 10;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */

  protected ClientInterface $httpClient;

  /**
   * The site's settings.
   *
   * @var \Drupal\Core\Site\Settings
   */

  protected Settings $siteSettings;

  /**
   * Drupal's entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */

  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity storage for TapisTokens.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */

  protected EntityStorageInterface $tokenStorage;

  /**
   * The TapisProvider for tapis sites/tenants.
   *
   * @var \Drupal\tapis_tenant\TapisProvider\TapisSiteTenantProviderInterface
   */

  protected TapisSiteTenantProviderInterface $tapisSiteTenantProvider;

  /**
   * The constructor.
   */
  public function __construct(
    ClientInterface $httpClient,
    Settings $siteSettings,
    EntityTypeManagerInterface $entityTypeManager,
    TapisSiteTenantProviderInterface $tapisSiteTenantProvider
  ) {
    $this->httpClient = $httpClient;
    $this->siteSettings = $siteSettings;
    $this->entityTypeManager = $entityTypeManager;
    $this->tokenStorage = $this->entityTypeManager->getStorage("tapis_token");
    $this->tapisSiteTenantProvider = $tapisSiteTenantProvider;
  }

  /**
   * Create instances.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('settings'),
      $container->get('entity_type.manager'),
      $container->get('tapis_tenant.tapis_site_tenant_provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getUserToken($tenantId, $uid) {
    // Check if the user can access Tapis services.
    if (!$this->canUserAccessTapisServices($tenantId, $uid)) {
      throw new AccessDeniedHttpException();
    }
    $ids = $this->tokenStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', TapisToken::TYPE_USER)
      ->condition("tenant", $tenantId)
      ->condition('uid', $uid)
      ->execute();
    if (!$ids) {
      $newToken = $this->getNewUserToken($tenantId, $uid);
      $this->tokenStorage->create([
        'type' => TapisToken::TYPE_USER,
        'uid' => ['target_id' => $uid],
        'tenant' => ['target_id' => $tenantId],
        'access_token' => $newToken['access_token'],
        'refresh_token' => $newToken['refresh_token'],
      ])->save();
      return $newToken['access_token'];
    }

    $tapisToken = $this->tokenStorage->load(array_shift($ids));
    /** @var \Drupal\tapis_auth\Entity\TapisToken $tapisToken */
    $accessToken = $tapisToken->getAccessToken();
    $accessTokenJWT = $this->decodeJWT($accessToken);

    // Check if the current timestamp
    // is within token expiry window.
    if (
      ($accessTokenJWT->exp - time()) <= self::TOKEN_EXPIRY_WINDOW_SECS
    ) {
      // Since the current token is about to expire,
      // get a new authenticator service token and save it.
      $newToken = $this->getNewUserToken($tenantId, $uid);
      $tapisToken->setAccessToken($newToken['access_token']);
      $tapisToken->setRefreshToken($newToken['refresh_token']);
      $tapisToken->save();

      // Delete all other existing tokens.
      foreach ($ids as $id) {
        $this->tokenStorage->load($id)->delete();
      }
      return $newToken['access_token'];
    }

    return $accessToken;
  }

  /**
   * {@inheritdoc}
   */
  public function canUserAccessTapisServices($tenantId, $uid) {
    if ($uid == 0) {
      // No tapis tokens for anon users.
      return FALSE;
    }

    $user = User::load($uid);

    // Check if the user has access to the Tapis tenant & site.
    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $tenant = $tenantInfo['tenant'];

    if (!$tenant->access('view', $user)) {
      // The user doesn't have access to the tenant.
      return FALSE;
    }

    // Check if the user has the "use tapis auth" permission.
    return $user->hasPermission(Constants::PERMISSION_USE_TAPIS_AUTH);
  }

  /**
   * Gets a new Tapis access token for a Drupal user.
   *
   * @param int $tenantId
   *   The Tapis tenant id.
   * @param int $uid
   *   The user id.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *    Thrown for unuthorized users or any API call to Tapis fails.
   *
   * @returns array A new Tapis access token for the user.
   *
   * @throws \Drupal\tapis_auth\Exception\TapisAuthException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  private function getNewUserToken($tenantId, $uid): array {
    // Check if the user can access Tapis services.
    if (!$this->canUserAccessTapisServices($tenantId, $uid)) {
      throw new AccessDeniedHttpException();
    }

    $username = $this->getTapisUsername($tenantId, $uid);

    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);

    // Use the authenticator service password to get a JWT
    // for the authenticator (which is our Drupal site)
    $tapis_site_id = $tenantInfo['tapis_site_id'];
    $tapis_tenant_id = $tenantInfo['tapis_id'];
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $authenticator_service = $tenantInfo['services']['authenticator']['service'];

    $authenticatorJWT = $this->getAuthenticatorServiceToken($tenantId);

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/tokens";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'headers' => [
        'X-Tapis-Token' => $authenticatorJWT,
        'X-Tapis-Tenant' => $tapis_tenant_id,
        'X-Tapis-User' => $authenticator_service,
      ],
      'json' => [
        "token_tenant_id" => $tapis_tenant_id,
        "target_site_id" => $tapis_site_id,
        'account_type' => 'user',
        'token_username' => $username,
        'access_token_ttl' => self::ACCESS_TOKEN_TTL,
        'refresh_token_ttl' => self::REFRESH_TOKEN_TTL,
        'generate_refresh_token' => TRUE,
      ],
      'http_errors' => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_auth')->debug(print_r($response_body, TRUE));

    // @todo Figure out how to handle Tapis api call errors
    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Tapis token for user id '$uid' in the '$tapis_tenant_id' tenant: $message");
    }

    if ($response_body['status'] !== "success") {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Tapis token for user id '$uid' in the '$tapis_tenant_id' tenant: $message");
    }

    return [
      'access_token' => $response_body['result']['access_token']['access_token'],
      'refresh_token' => $response_body['result']['refresh_token']['refresh_token'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getTapisUsername($tenantId, $uid): string {
    // NOTE: We use this as the username within Tapis
    // instead of the user's actual unique username within Drupal,
    // since the user can always change their username within Drupal,
    // but we can't do this within Tapis.
    // Also, in the current design,
    // the tapis username for a user doesn't depend on the tenant id.
    if (!$this->canUserAccessTapisServices($tenantId, $uid)) {
      // No tapis tokens for unauthorized users.
      throw new AccessDeniedHttpException();
    }
    return "drupaluid_$uid";
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthenticatorServiceToken($tenantId) {
    $ids = $this->tokenStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', TapisToken::TYPE_SERVICE)
      ->condition("tenant", $tenantId)
      ->condition('service', 'authenticator')
      ->execute();
    if (!$ids) {
      $newToken = $this->getNewAuthenticatorServiceToken($tenantId);
      $this->tokenStorage->create([
        'type' => TapisToken::TYPE_SERVICE,
        'tenant' => ['target_id' => $tenantId],
        'service' => 'authenticator',
        'access_token' => $newToken['access_token'],
        'refresh_token' => $newToken['refresh_token'],
      ])->save();
      return $newToken['access_token'];
    }
    $tapisToken = $this->tokenStorage->load(array_shift($ids));

    /** @var \Drupal\tapis_auth\Entity\TapisToken $tapisToken */
    $accessToken = $tapisToken->getAccessToken();
    $accessTokenJWT = $this->decodeJWT($accessToken);

    // Check if the current timestamp
    // is within the token expiry window.
    if (($accessTokenJWT->exp - time()) <= self::TOKEN_EXPIRY_WINDOW_SECS) {
      // Since the current token is about to expire,
      // get a new authenticator service token and save it.
      $newToken = $this->getNewAuthenticatorServiceToken($tenantId);
      $tapisToken->setAccessToken($newToken['access_token']);
      $tapisToken->setRefreshToken($newToken['refresh_token']);
      $tapisToken->save();

      // Delete all other existing tokens.
      foreach ($ids as $id) {
        $this->tokenStorage->load($id)->delete();
      }
      return $newToken['access_token'];
    }

    return $accessToken;
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_auth\Exception\TapisAuthException
   */
  public function getNewAuthenticatorServiceToken($tenantId): array {
    // Use the authenticator service password to get a JWT
    // for the authenticator (which is our Drupal site)
    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);

    $tapis_site_id = $tenantInfo['tapis_site_id'];
    $tapis_tenant_id = $tenantInfo['tapis_site_admin_tenant_id'];
    $tapis_api_endpoint = $tenantInfo['tapis_site_admin_tenant_api_endpoint'];

    $authenticator_service = $tenantInfo['services']['authenticator']['service'];
    $authenticator_service_password = $tenantInfo['services']['authenticator']['service_password'];
    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/tokens";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'auth' => [
        $authenticator_service,
        $authenticator_service_password,
      ],
      'json' => [
        "token_tenant_id" => $tapis_tenant_id,
        'account_type' => 'service',
        'token_username' => $authenticator_service,
        "target_site_id" => $tapis_site_id,
        'access_token_ttl' => self::ACCESS_TOKEN_TTL,
        'refresh_token_ttl' => self::REFRESH_TOKEN_TTL,
        'generate_refresh_token' => TRUE,
      ],
      'http_errors' => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_auth')->debug(print_r($response_body, TRUE));

    // @todo Figure out how to handle Tapis api call errors
    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Tapis authenticator service token in the '$tapis_tenant_id' admin tenant: $message");
    }

    if ($response_body['status'] !== "success") {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Tapis authenticator service token in the '$tapis_tenant_id' admin tenant: $message");
    }

    return [
      'access_token' => $response_body['result']['access_token']['access_token'],
      'refresh_token' => $response_body['result']['refresh_token']['refresh_token'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function decodeJWT($jwt) {
    return json_decode(base64_decode(str_replace('_', '/', str_replace('-', '+', explode('.', $jwt)[1]))));
  }

  /**
   * {@inheritdoc}
   */
  public function validateTenantAuthSettings(
    $tapis_tenant_id,
    $tapis_site_id,
    $tapis_api_endpoint,
    $authenticator_service,
    $authenticator_service_password
  ): array {
    // Use the authenticator service password to get a JWT
    // for the authenticator (which is our Drupal site)
    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/tokens";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'auth' => [
        $authenticator_service,
        $authenticator_service_password,
      ],
      'json' => [
        "token_tenant_id" => $tapis_tenant_id,
        'account_type' => 'service',
        'token_username' => $authenticator_service,
        "target_site_id" => $tapis_site_id,
        'access_token_ttl' => self::ACCESS_TOKEN_TTL,
        'refresh_token_ttl' => self::REFRESH_TOKEN_TTL,
        'generate_refresh_token' => TRUE,
      ],
      'http_errors' => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_auth')->debug(print_r($response_body, TRUE));

    // @todo Figure out how to handle Tapis api call errors
    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Tapis authenticator service token in the '$tapis_tenant_id' admin tenant: $message");
    }

    if ($response_body['status'] !== "success") {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Tapis authenticator service token in the '$tapis_tenant_id' admin tenant: $message");
    }

    return [
      'access_token' => $response_body['result']['access_token']['access_token'],
      'refresh_token' => $response_body['result']['refresh_token']['refresh_token'],
    ];
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\tapis_auth\Exception\TapisAuthException
   */
  public function getIntegrationTestUserToken($tenantId, $tapis_username) {
    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);

    // Use the authenticator service password to get a JWT
    // for the authenticator (which is our Drupal site)
    $tapis_site_id = $tenantInfo['tapis_site_id'];
    $tapis_tenant_id = $tenantInfo['tapis_id'];
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];
    $authenticator_service = $tenantInfo['services']['authenticator']['service'];

    $authenticatorJWT = $this->getAuthenticatorServiceToken($tenantId);

    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/tokens";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'headers' => [
        'X-Tapis-Token' => $authenticatorJWT,
        'X-Tapis-Tenant' => $tapis_tenant_id,
        'X-Tapis-User' => $authenticator_service,
      ],
      'json' => [
        "token_tenant_id" => $tapis_tenant_id,
        "target_site_id" => $tapis_site_id,
        'account_type' => 'user',
        'token_username' => $tapis_username,
        'access_token_ttl' => self::ACCESS_TOKEN_TTL,
        'refresh_token_ttl' => self::REFRESH_TOKEN_TTL,
        'generate_refresh_token' => TRUE,
      ],
      'http_errors' => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_auth')->debug(print_r($response_body, TRUE));

    // @todo Figure out how to handle Tapis api call errors
    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Tapis token for user id '$tapis_username' in the '$tapis_tenant_id' tenant: $message");
    }

    if ($response_body['status'] !== "success") {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Tapis token for user id '$tapis_username' in the '$tapis_tenant_id' tenant: $message");
    }

    return [
      'access_token' => $response_body['result']['access_token']['access_token'],
      'refresh_token' => $response_body['result']['refresh_token']['refresh_token'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getDrupalServiceToken($tenantId) {
    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);
    $service_username = $tenantInfo['service_account_username'];

    $ids = $this->tokenStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', TapisToken::TYPE_SERVICE)
      ->condition('tenant', $tenantId)
      ->condition('service', $service_username)
      ->execute();
    if (!$ids) {
      $newToken = $this->getNewDrupalServiceToken($tenantId);
      $this->tokenStorage->create([
        'type' => TapisToken::TYPE_SERVICE,
        'tenant' => ['target_id' => $tenantId],
        'service' => $service_username,
        'access_token' => $newToken['access_token'],
        'refresh_token' => $newToken['refresh_token'],
      ])->save();
      return $newToken['access_token'];
    }

    $tapisToken = $this->tokenStorage->load(array_shift($ids));
    /** @var \Drupal\tapis_auth\Entity\TapisToken $tapisToken */
    $accessToken = $tapisToken->getAccessToken();
    $accessTokenJWT = $this->decodeJWT($accessToken);

    // Check if the current timestamp
    // is within the token expiry window.
    if (($accessTokenJWT->exp - time()) <= self::TOKEN_EXPIRY_WINDOW_SECS) {

      // Since the current token is about to expire,
      // get a new authenticator service token and save it.
      $newToken = $this->getNewDrupalServiceToken($tenantId);
      $tapisToken->setAccessToken($newToken['access_token']);
      $tapisToken->setRefreshToken($newToken['refresh_token']);
      $tapisToken->save();

      // Delete all other existing tokens.
      foreach ($ids as $id) {
        $this->tokenStorage->load($id)->delete();
      }

      return $newToken['access_token'];
    }
    return $accessToken;
  }

  /**
   * Gets a new Tapis service access token for Drupal.
   *
   * @param int $tenantId
   *   The Tapis tenant id.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *    Thrown when any API call to Tapis fails.
   *
   * @returns array A new Tapis service access token for Drupal.
   *
   * @throws \Drupal\tapis_auth\Exception\TapisAuthException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  private function getNewDrupalServiceToken($tenantId): array {
    // Use the authenticator service password
    // to get a JWT for the authenticator (which is our Drupal site)
    $tenantInfo = $this->tapisSiteTenantProvider->getTenantInfo($tenantId);

    $service_username = $tenantInfo['service_account_username'];
    $tapis_site_id = $tenantInfo['tapis_site_id'];
    $tapis_tenant_id = $tenantInfo['tapis_id'];
    $tapis_api_endpoint = $tenantInfo['tapis_api_endpoint'];

    $authenticator_service = $tenantInfo['services']['authenticator']['service'];
    $authenticatorJWT = $this->getAuthenticatorServiceToken($tenantId);
    $tapis_api_url = "$tapis_api_endpoint/" . self::TAPIS_API_VERSION . "/tokens";

    $response = $this->httpClient->request('POST', $tapis_api_url, [
      'headers' => [
        'X-Tapis-Token' => $authenticatorJWT,
        'X-Tapis-Tenant' => $tapis_tenant_id,
        'X-Tapis-User' => $authenticator_service,
      ],
      'json' => [
        "target_site_id" => $tapis_site_id,
        "token_tenant_id" => $tapis_tenant_id,
        'account_type' => 'user',
        'token_username' => $service_username,
        'access_token_ttl' => self::ACCESS_TOKEN_TTL,
        'refresh_token_ttl' => self::REFRESH_TOKEN_TTL,
        'generate_refresh_token' => TRUE,
      ],
      'http_errors' => FALSE,
    ]);

    $response_body = json_decode($response->getBody(), TRUE);

    // Log the response array as a string.
    \Drupal::logger('tapis_auth')->debug(print_r($response_body, TRUE));

    // @todo Figure out how to handle Tapis api call errors
    if ($response->getStatusCode() !== 200) {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Drupal service service token in the '$tapis_tenant_id' tenant: $message");
    }

    if ($response_body['status'] !== "success") {
      if (is_array($response_body) && isset($response_body['message'])) {
        $message = $response_body['message'];
      } else {
        // Optionally, include the raw body or a default message
        $message = $response->getBody()->getContents() ?: 'No error message returned from API.';
      }
      throw new TapisAuthException("Could not fetch a new Drupal service service token in the '$tapis_tenant_id' tenant: $message");
    }

    return [
      'access_token' => $response_body['result']['access_token']['access_token'],
      'refresh_token' => $response_body['result']['refresh_token']['refresh_token'],
    ];
  }

}
